package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDetach(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		detach, err := NewDetach(protocol.FirstRequestID, addr)
		require.NoError(t, err)
		assert.Equal(t, protocol.Detach, detach.OpCode())
		assert.Equal(t, protocol.FirstRequestID, detach.RequestID())
		assert.Equal(t, addr, detach.Address())
		assert.Equal(t, "<detach> reqID=1, addr=test@1", detach.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		detach, err := NewDetach(protocol.FirstRequestID, addr)
		require.NoError(t, err)
		bytes, err := detach.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, detach, out)
	})

	t.Run("should report detach id errors", func(t *testing.T) {
		_, err := NewDetach(protocol.NoRequestID, addr)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&detachMessage{protocol.NoRequestID, addr}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Detach).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Detach).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Detach).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewDetach(protocol.FirstRequestID, nil)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&detachMessage{protocol.FirstRequestID, &badAddr{}}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Detach).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(detachAddressOffset, 0),
			createBlank(protocol.Detach).Unmarshal(protocol.Current, []byte{}))
	})
}
