package message

import (
	"encoding/binary"
	"fmt"
	"sort"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	detachedRequestIDOffset = headerLength
	detachedScopesOffset    = detachedRequestIDOffset + 4
)

// TODO : report source ids to allow consistent and fair detach logic
// Detached reports a successful allocate attempt
type Detached interface {
	protocol.Response
	Scopes() stream.AddressScopes
}

type detachedMessage struct {
	id     protocol.RequestID
	scopes stream.AddressScopes
}

// NewDetached creates an Detached
func NewDetached(id protocol.RequestID, scopes stream.AddressScopes) (Detached, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	sorted := make(stream.AddressScopes, len(scopes))
	copy(sorted, scopes)
	sort.Sort(sorted)
	return &detachedMessage{id, sorted}, nil
}

func (*detachedMessage) OpCode() protocol.OpCode            { return protocol.Detached }
func (a *detachedMessage) ForRequestID() protocol.RequestID { return a.id }
func (a *detachedMessage) Scopes() stream.AddressScopes     { return a.scopes }

func (a *detachedMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}

	data, _ := a.scopes.MarshalBinary()
	bytes := make([]byte, allocateAddressOffset+len(data))
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint32(bytes[detachedRequestIDOffset:], uint32(a.id))
	copy(bytes[detachedScopesOffset:], data)

	return bytes, nil
}

func (a *detachedMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < detachedScopesOffset {
		return protocol.ErrInvalidLength(detachedScopesOffset, total)
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[detachedRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	var scopes stream.AddressScopes
	if err := scopes.UnmarshalBinary(bytes[detachedScopesOffset:]); err != nil {
		return err
	}
	a.id = id
	a.scopes = scopes
	return nil
}

func (a *detachedMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, scopes=%s", a.OpCode(), a.ForRequestID(), a.Scopes())
}
