package message

import (
	"sort"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDetached(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)
	scopes := stream.AddressScopes{stream.AnyAddress, addr}

	t.Run("should report values correctly", func(t *testing.T) {
		detached, err := NewDetached(protocol.FirstRequestID, scopes)
		require.NoError(t, err)
		assert.Equal(t, protocol.Detached, detached.OpCode())
		assert.Equal(t, protocol.FirstRequestID, detached.ForRequestID())
		assert.NotEqual(t, scopes, detached.Scopes()) // due to resort
		sorted := make(stream.AddressScopes, len(scopes))
		copy(sorted, scopes)
		sort.Sort(sorted)
		assert.Equal(t, sorted, detached.Scopes())
		assert.Equal(t, "<detached> reqID=1, scopes=[^test@1 ^*]", detached.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		detached, err := NewDetached(protocol.FirstRequestID, scopes)
		require.NoError(t, err)
		bytes, err := detached.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, detached, out)
	})

	t.Run("should report detached id errors", func(t *testing.T) {
		_, err := NewDetached(protocol.NoRequestID, scopes)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&detachedMessage{protocol.NoRequestID, scopes}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Detached).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Detached).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Detached).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		msg, _ := NewDetached(protocol.FirstRequestID, stream.AddressScopes{&badAddr{}})
		bytes, err := msg.Marshal(protocol.Current)
		require.NoError(t, err) // marshal doesn't validate
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Detached).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(detachedScopesOffset, 0),
			createBlank(protocol.Detached).Unmarshal(protocol.Current, []byte{}))
	})
}
