package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/errors"
)

const (
	errorAckIDOffset   = headerLength
	errorMessageOffset = errorAckIDOffset + 2
)

type Error interface {
	protocol.Message
	ForAckID() protocol.AckID
	Unwrap() error
}

type errorMessage struct {
	forId protocol.AckID
	value error
}

// NewError creates an Error - note that it accepts NoRequest
func NewError(id protocol.AckID, value error) (Error, error) {
	if value == nil {
		value = protocol.ErrServiceTimedOut
	}
	return &errorMessage{forId: id, value: value}, nil
}

func (*errorMessage) OpCode() protocol.OpCode    { return protocol.Error }
func (e *errorMessage) ForAckID() protocol.AckID { return e.forId }
func (e *errorMessage) Unwrap() error            { return e.value }
func (e *errorMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	msg, err := protocol.Errors.Marshal(e.value)
	if err != nil {
		return nil, err
	}
	bytes := make([]byte, errorMessageOffset, errorMessageOffset+len(msg))
	injectHeader(bytes, ver, e.OpCode())
	binary.BigEndian.PutUint16(bytes[errorAckIDOffset:], uint16(e.forId))
	return append(bytes, msg...), nil
}

func (a *errorMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < errorMessageOffset {
		return protocol.ErrInvalidLength(errorMessageOffset, len(bytes))
	}

	id := protocol.AckID(binary.BigEndian.Uint16(bytes[errorAckIDOffset:]))
	if err := protocol.Errors.Unmarshal(bytes[errorMessageOffset:], &a.value); err != nil {
		return err
	}
	a.forId = id
	return nil
}

func (e *errorMessage) String() string {
	err := e.Unwrap()
	if errC, ok := err.(errors.ErrorCodeError); ok {
		return fmt.Sprintf("<%s> reqID=%d, code=%s, err=%v", e.OpCode(), e.ForAckID(), errC.ErrorCode(), errC.Error())
	}
	return fmt.Sprintf("<%s> reqID=%d, err=%v", e.OpCode(), e.ForAckID(), err)
}
