package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
)

const (
	failureRequestIDOffset = headerLength
	failureMessageOffset   = failureRequestIDOffset + 2
	failureDefaultMessage  = "Service error"
)

type Failure interface {
	protocol.Response
	error
}

type failureMessage struct {
	forId protocol.RequestID
	msg   string
}

// NewFailure creates a Failure
func NewFailure(id protocol.RequestID, err error) (Failure, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	msg := failureDefaultMessage
	if err != nil {
		msg = err.Error()
	}
	return &failureMessage{id, msg}, nil
}

func (*failureMessage) OpCode() protocol.OpCode            { return protocol.Failure }
func (f *failureMessage) ForRequestID() protocol.RequestID { return f.forId }
func (f *failureMessage) Error() string                    { return f.msg }
func (f *failureMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, errorMessageOffset, errorMessageOffset+len(f.msg))
	injectHeader(bytes, ver, f.OpCode())
	binary.BigEndian.PutUint16(bytes[failureRequestIDOffset:], uint16(f.forId))
	return append(bytes, f.msg...), nil
}

func (f *failureMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < failureMessageOffset {
		return protocol.ErrInvalidLength(failureMessageOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[failureRequestIDOffset:]))
	f.forId = id
	f.msg = string(bytes[failureMessageOffset:])
	return nil
}

func (f *failureMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, err=%s", f.OpCode(), f.ForRequestID(), f.Error())
}
