package message

import (
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	prepareAddressOffset = headerLength
)

// Preprare a proposal for paxos single-degree proposes a source for an address
// to the cluster
type Prepare interface {
	protocol.Message
	Address() stream.Address // scope of election
	ID() protocol.SuggestionID
}

type prepareMessage struct {
	addr stream.Address
	id   protocol.SuggestionID
}

func NewPrepare(addr stream.Address, id protocol.SuggestionID) (Prepare, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &prepareMessage{addr, id}, nil
}

func (*prepareMessage) OpCode() protocol.OpCode     { return protocol.Prepare }
func (p *prepareMessage) ID() protocol.SuggestionID { return p.id }
func (p *prepareMessage) Address() stream.Address   { return p.addr }

func (p *prepareMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	if !p.id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	key := stream.NoAddress
	if p.addr != nil {
		key = p.addr.Key()
	}
	suggestionOffset := prepareAddressOffset + len(key)
	bytes := make([]byte, suggestionOffset+1+p.id.Len())
	injectHeader(bytes, ver, p.OpCode())
	copy(bytes[prepareAddressOffset:], key)
	bytes[suggestionOffset] = '\000'
	return bytes, p.id.Marshal(bytes[suggestionOffset+1:])
}

func (p *prepareMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < prepareAddressOffset {
		return protocol.ErrInvalidLength(prepareAddressOffset, total)
	}

	suggestionOffset := prepareAddressOffset
	for suggestionOffset < total {
		if bytes[suggestionOffset] == '\000' {
			break
		}
		suggestionOffset++
	}

	if suggestionOffset == total {
		return protocol.ErrEOF
	}

	addr, err := stream.ParseAddress(string(bytes[prepareAddressOffset:suggestionOffset]))
	if err != nil {
		return err
	}
	p.addr = addr
	p.id, _ = protocol.UnmarshalSuggestion(bytes[suggestionOffset+1:])
	return nil
}

func (p *prepareMessage) String() string {
	return fmt.Sprintf("<%v> suggestion=%v, addr=%s", p.OpCode(), p.ID(), p.Address().Key())
}
