package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestPrepare(t *testing.T) {
	sid := protocol.FirstSuggestion("source")
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		request, err := NewPrepare(addr, sid)
		require.NoError(t, err)
		assert.Equal(t, protocol.Prepare, request.OpCode())
		assert.Equal(t, sid, request.ID())
		assert.Equal(t, addr, request.Address())
		assert.Equal(t, "<prepare> suggestion=source#0, addr=test@1", request.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		request, err := NewPrepare(addr, sid)
		require.NoError(t, err)
		bytes, err := request.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, request, out)
	})

	t.Run("should report suggestion id errors", func(t *testing.T) {
		_, err := NewPrepare(addr, protocol.NoSuggestion)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
		_, err = (&prepareMessage{addr, protocol.NoSuggestion}).Marshal(protocol.Current)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Prepare).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Prepare).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewPrepare(nil, sid)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&prepareMessage{&badAddr{}, sid}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Prepare).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(prepareAddressOffset, 0),
			createBlank(protocol.Prepare).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, prepareAddressOffset)
		injectHeader(broken, protocol.One, protocol.Prepare)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Prepare).Unmarshal(protocol.Current, broken))
	})
}
