package message

import (
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	promiseAddressOffset = headerLength
)

// Promise that a value with the given id will be accepted
type Promise interface {
	protocol.Message
	ID() protocol.SuggestionID
	Address() stream.Address
	Accepted() (protocol.Proposal, bool)
}

type promiseMessage struct {
	addr     stream.Address
	id       protocol.SuggestionID
	proposal protocol.Proposal
}

func NewPromise(addr stream.Address, id protocol.SuggestionID, proposed protocol.Proposal) (Promise, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &promiseMessage{addr, id, proposed}, nil
}

func (*promiseMessage) OpCode() protocol.OpCode               { return protocol.Promise }
func (p *promiseMessage) ID() protocol.SuggestionID           { return p.id }
func (p *promiseMessage) Address() stream.Address             { return p.addr }
func (p *promiseMessage) Accepted() (protocol.Proposal, bool) { return p.proposal, p.proposal != nil }

func (p *promiseMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	if !p.id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	key := stream.NoAddress
	if p.addr != nil {
		key = p.addr.Key()
	}

	suggestionOffset := prepareAddressOffset + len(key)
	hostOffset := suggestionOffset + 1 + p.id.Len()
	hostLength := 0
	if p.proposal != nil {
		hostLength = len(p.proposal.Hostname())
	}
	bytes := make([]byte, hostOffset+1+hostLength)
	injectHeader(bytes, ver, p.OpCode())
	copy(bytes[promiseAddressOffset:], key)
	bytes[suggestionOffset] = '\000'
	err := p.id.Marshal(bytes[suggestionOffset+1:])
	bytes[hostOffset] = '\000'
	if p.proposal != nil {
		copy(bytes[hostOffset+1:], p.proposal.Hostname())
	}
	return bytes, err
}

func (p *promiseMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < promiseAddressOffset {
		return protocol.ErrInvalidLength(promiseAddressOffset, total)
	}

	suggestionOffset := promiseAddressOffset
	for suggestionOffset < total {
		if bytes[suggestionOffset] == '\000' {
			break
		}
		suggestionOffset++
	}

	if suggestionOffset == total {
		return protocol.ErrEOF
	}

	hostOffset := suggestionOffset + 5
	for hostOffset < total {
		if bytes[hostOffset] == '\000' {
			break
		}
		hostOffset++
	}

	if hostOffset >= total {
		return protocol.ErrEOF
	}

	host := string(bytes[hostOffset+1:])

	addr, err := stream.ParseAddress(string(bytes[promiseAddressOffset:suggestionOffset]))
	if err != nil {
		return err
	}

	id, _ := protocol.UnmarshalSuggestion(bytes[suggestionOffset+1 : hostOffset])

	var proposal protocol.Proposal
	if host != "" {
		proposal, err = NewAccepted(addr, id, host)
		if err != nil {
			return err
		}
	}

	p.id = id
	p.addr = addr
	p.proposal = proposal
	return nil
}

func (p *promiseMessage) String() string {
	var host string
	if p.proposal != nil {
		host = p.proposal.Hostname()
	}
	return fmt.Sprintf("<%v> suggestion=%v, addr=%s, host=%s", p.OpCode(), p.ID(), p.Address().Key(), host)
}
