package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestPromise(t *testing.T) {
	sid := protocol.FirstSuggestion("source")
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)
	proposal, errx := NewAccepted(addr, sid, "host")
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		request, err := NewPromise(addr, sid, proposal)
		require.NoError(t, err)
		assert.Equal(t, protocol.Promise, request.OpCode())
		assert.Equal(t, sid, request.ID())
		assert.Equal(t, addr, request.Address())
		accepted, exists := request.Accepted()
		assert.True(t, exists)
		assert.Equal(t, proposal, accepted)
		assert.Equal(t, "<promise> suggestion=source#0, addr=test@1, host=host", request.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		request, err := NewPromise(addr, sid, proposal)
		require.NoError(t, err)
		bytes, err := request.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, request, out)
	})

	t.Run("should report suggestion id errors", func(t *testing.T) {
		_, err := NewPromise(addr, protocol.NoSuggestion, nil)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
		_, err = (&promiseMessage{addr, protocol.NoSuggestion, nil}).Marshal(protocol.Current)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Promise).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Promise).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewPromise(nil, sid, nil)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&promiseMessage{&badAddr{}, sid, nil}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Promise).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should forward proposal host errors", func(t *testing.T) {
		promise, err := NewPromise(addr, sid, &acceptedMessage{addr, sid, "::?"})
		require.NoError(t, err)
		bytes, err := promise.Marshal(protocol.Current)
		require.NoError(t, err)
		_, err = Unmarshal(bytes)
		assert.Equal(t, protocol.ErrInvalidHost, err)
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(promiseAddressOffset, 0),
			createBlank(protocol.Promise).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, promiseAddressOffset)
		injectHeader(broken, protocol.Current, protocol.Promise)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Promise).Unmarshal(protocol.Current, broken))
		broken = append(broken, "\000source"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Promise).Unmarshal(protocol.Current, broken))
	})
}
