package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	reserveRequestIDOffset = headerLength
	reserveAddressOffset   = reserveRequestIDOffset + 4
)

// Reserve attempts to reserve a connection slot
type Reserve interface {
	protocol.Request
	Address() stream.Address
	Credentials() stream.Credentials
}

type reserveMessage struct {
	id    protocol.RequestID
	addr  stream.Address
	creds stream.Credentials
}

// NewReserve creates a Reserve
func NewReserve(id protocol.RequestID, addr stream.Address, creds stream.Credentials) (Reserve, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	if creds == nil {
		return nil, protocol.ErrMissingCredentials
	}
	return &reserveMessage{id, addr, creds}, nil
}

func (*reserveMessage) OpCode() protocol.OpCode           { return protocol.Reserve }
func (r *reserveMessage) RequestID() protocol.RequestID   { return r.id }
func (r *reserveMessage) Address() stream.Address         { return r.addr }
func (r *reserveMessage) Credentials() stream.Credentials { return r.creds }

func (r *reserveMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if r.addr != nil {
		key = r.addr.Key()
	}
	cbytes, _ := r.creds.MarshalBinary()
	bytes := make([]byte, reserveAddressOffset+len(key)+1+len(cbytes))
	injectHeader(bytes, ver, r.OpCode())
	binary.BigEndian.PutUint32(bytes[reserveRequestIDOffset:], uint32(r.id))
	copy(bytes[reserveAddressOffset:], key)

	authOffset := reserveAddressOffset + len(key)
	bytes[authOffset] = '\000'
	copy(bytes[authOffset+1:], cbytes)

	return bytes, nil
}

func (r *reserveMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < reserveAddressOffset {
		return protocol.ErrInvalidLength(reserveAddressOffset, total)
	}
	authOffset := reserveAddressOffset
	for authOffset < total {
		if bytes[authOffset] == '\000' {
			break
		}
		authOffset++
	}
	if authOffset == total {
		return protocol.ErrEOF
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[reserveRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	addr, err := stream.ParseAddress(string(bytes[reserveAddressOffset:authOffset]))
	if err != nil {
		return err
	}
	creds, err := stream.UnmarshalCredentials(bytes[authOffset+1:])
	if err != nil {
		return err
	}
	r.id = id
	r.addr = addr
	r.creds = creds
	return nil
}

func (r *reserveMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, addr=%s, creds=%s", r.OpCode(), r.RequestID(), r.Address().Key(), r.Credentials().ClientID())
}
