package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	scopesAckIDOffset   = headerLength
	scopesFlagsOffset   = scopesAckIDOffset + 2
	scopesContentOffset = scopesFlagsOffset + 1
)

type Scopes interface {
	protocol.Ackable
	Scopes() stream.AddressSourceMap
	Remove() bool
}

type scopesMessage struct {
	id     protocol.AckID
	remove bool
	scopes stream.AddressSourceMap
}

// NewScopes details how host support for scopes has changed
func NewScopes(id protocol.AckID, scopes stream.AddressSourceMap, remove bool) (Scopes, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidAckID
	}
	copied := make(stream.AddressSourceMap, len(scopes))
	if scopes != nil {
		for k, v := range scopes {
			copied[k] = v
		}
	}
	return &scopesMessage{id, remove, copied}, nil
}

func (*scopesMessage) OpCode() protocol.OpCode           { return protocol.Scopes }
func (s *scopesMessage) AckID() protocol.AckID           { return s.id }
func (s *scopesMessage) Scopes() stream.AddressSourceMap { return s.scopes }
func (s *scopesMessage) Remove() bool                    { return s.remove }
func (s *scopesMessage) Marshal(ver protocol.Version) ([]byte, error) {
	switch ver {
	case protocol.One:
		return s.marshalOne()
	case protocol.Two:
		return s.marshalTwo()
	default:
		return nil, protocol.ErrInvalidVersion
	}
}

func (s *scopesMessage) marshalOne() ([]byte, error) {
	array, err := s.scopes.ToScopes()
	if err != nil {
		return nil, err
	}
	scopes, _ := array.MarshalBinary()
	bytes := make([]byte, scopesContentOffset+len(scopes))
	injectHeader(bytes, protocol.One, s.OpCode())
	binary.BigEndian.PutUint16(bytes[scopesAckIDOffset:], uint16(s.id))
	if s.remove {
		bytes[scopesFlagsOffset] = 1
	} else {
		bytes[scopesFlagsOffset] = 0
	}
	copy(bytes[scopesContentOffset:], scopes)
	return bytes, nil
}

func (s *scopesMessage) marshalTwo() ([]byte, error) {
	scopes, _ := s.scopes.MarshalBinary()
	bytes := make([]byte, scopesContentOffset+len(scopes))
	injectHeader(bytes, protocol.Two, s.OpCode())
	binary.BigEndian.PutUint16(bytes[scopesAckIDOffset:], uint16(s.id))
	if s.remove {
		bytes[scopesFlagsOffset] = 1
	} else {
		bytes[scopesFlagsOffset] = 0
	}
	copy(bytes[scopesContentOffset:], scopes)
	return bytes, nil
}

func (s *scopesMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	switch ver {
	case protocol.One:
		return s.unmarshalOne(bytes)
	case protocol.Two:
		return s.unmarshalTwo(bytes)
	default:
		return protocol.ErrInvalidVersion
	}
}

func (s *scopesMessage) unmarshalOne(bytes []byte) error {
	total := len(bytes)
	if total < scopesContentOffset {
		return protocol.ErrInvalidLength(scopesContentOffset, total)
	}
	id := protocol.AckID(binary.BigEndian.Uint16(bytes[scopesAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}
	var scopes stream.AddressScopes
	if err := scopes.UnmarshalBinary(bytes[scopesContentOffset:]); err != nil {
		return err
	}

	s.remove = false
	if bytes[scopesFlagsOffset] != 0 {
		s.remove = true
	}

	s.id = id
	s.scopes = make(stream.AddressSourceMap)
	for _, k := range scopes {
		s.scopes[k.Key()] = 0
	}
	return nil
}

func (s *scopesMessage) unmarshalTwo(bytes []byte) error {
	total := len(bytes)
	if total < scopesContentOffset {
		return protocol.ErrInvalidLength(scopesContentOffset, total)
	}
	id := protocol.AckID(binary.BigEndian.Uint16(bytes[scopesAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}
	var scopes stream.AddressSourceMap
	if err := scopes.UnmarshalBinary(bytes[scopesContentOffset:]); err != nil {
		return err
	}

	s.remove = false
	if bytes[scopesFlagsOffset] != 0 {
		s.remove = true
	}

	s.id = id
	s.scopes = scopes
	return nil
}

func (s *scopesMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, remove=%v, scopes=%v", s.OpCode(), s.AckID(), s.Remove(), s.Scopes())
}
