package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
)

const (
	statusFlagsOffset      = headerLength
	statusLoadFactorOffset = statusFlagsOffset + 1
	statusLength           = statusLoadFactorOffset + 8
)

var (
	StatusUnknown, _ = NewStatus(protocol.NoLoad, protocol.NoFlags)
	StatusClosed, _  = NewStatus(protocol.NoLoad, protocol.Draining)
)

type Status interface {
	protocol.Message
	Flags() protocol.StatusFlags
	LoadFactor() protocol.LoadFactor
}

type statusMessage struct {
	load  protocol.LoadFactor
	flags protocol.StatusFlags
}

func NewStatus(load protocol.LoadFactor, flags protocol.StatusFlags) (Status, error) {
	return &statusMessage{load, flags}, nil
}

func (*statusMessage) OpCode() protocol.OpCode           { return protocol.Status }
func (s *statusMessage) LoadFactor() protocol.LoadFactor { return s.load }
func (s *statusMessage) Flags() protocol.StatusFlags     { return s.flags }
func (s *statusMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, statusLength)
	injectHeader(bytes, ver, s.OpCode())
	binary.BigEndian.PutUint64(bytes[statusLoadFactorOffset:], uint64(s.load))
	bytes[statusFlagsOffset] = byte(s.flags)
	return bytes, nil
}

func (s *statusMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total != statusLength {
		return protocol.ErrInvalidLength(statusLength, total)
	}
	s.load = protocol.LoadFactor(binary.BigEndian.Uint64(bytes[statusLoadFactorOffset:]))
	s.flags = protocol.StatusFlags(bytes[statusFlagsOffset])
	return nil
}

func (s *statusMessage) String() string {
	return fmt.Sprintf("<%v> flags=[%v], load=%v", s.OpCode(), s.Flags(), s.LoadFactor())
}
