package message

import (
	"encoding/binary"
	"fmt"
	"sort"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	ticketRequestIDOffset = headerLength
	ticketCodeOffset      = ticketRequestIDOffset + 4
)

// Ticket is the response to Reserve on success or failure
type Ticket interface {
	protocol.Response
	AccessCode() OpaqueBytes
	Scopes() stream.AddressScopes // may be wider than requested as an optimization
}

type ticketMessage struct {
	id     protocol.RequestID
	code   OpaqueBytes
	scopes stream.AddressScopes
}

// NewTicket creates a Ticket
func NewTicket(id protocol.RequestID, code OpaqueBytes, scopes stream.AddressScopes) (Ticket, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	return &ticketMessage{id, code, scopes}, nil
}

func (*ticketMessage) OpCode() protocol.OpCode            { return protocol.Ticket }
func (t *ticketMessage) ForRequestID() protocol.RequestID { return t.id }
func (t *ticketMessage) AccessCode() OpaqueBytes          { return t.code }
func (t *ticketMessage) Scopes() stream.AddressScopes     { return t.scopes }

func (t *ticketMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	scopes, _ := t.scopes.MarshalBinary()
	bytes := make([]byte, ticketCodeOffset+1+len(t.code)+len(scopes))
	injectHeader(bytes, ver, t.OpCode())
	binary.BigEndian.PutUint32(bytes[ticketRequestIDOffset:], uint32(t.id))

	copy(bytes[ticketCodeOffset:], t.code)
	scopesOffset := ticketCodeOffset + len(t.code)
	bytes[scopesOffset] = '\000'
	copy(bytes[scopesOffset+1:], scopes)

	return bytes, nil
}

func (t *ticketMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < ticketCodeOffset {
		return protocol.ErrInvalidLength(ticketCodeOffset, len(bytes))
	}

	scopesOffset := ticketCodeOffset
	for scopesOffset < total {
		if bytes[scopesOffset] == '\000' {
			break
		}
		scopesOffset++
	}

	if scopesOffset >= total {
		return protocol.ErrEOF
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[ticketRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}

	var scopes stream.AddressScopes
	if err := scopes.UnmarshalBinary(bytes[scopesOffset+1:]); err != nil {
		return err
	}
	sort.Sort(scopes)

	t.id = id
	t.code = OpaqueBytes(bytes[ticketCodeOffset:scopesOffset])
	t.scopes = scopes

	return nil
}

func (t *ticketMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, ####, scopes=%s", t.OpCode(), t.ForRequestID(), t.Scopes())
}
