package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestTicket(t *testing.T) {
	code := OpaqueBytes("Admit One")
	scopes := stream.AddressScopes{stream.AnyAddress}

	t.Run("should report no error correctly", func(t *testing.T) {
		request, err := NewTicket(protocol.FirstRequestID, code, scopes)
		require.NoError(t, err)
		assert.Equal(t, protocol.Ticket, request.OpCode())
		assert.Equal(t, protocol.FirstRequestID, request.ForRequestID())
		assert.Equal(t, scopes, request.Scopes())
		assert.Equal(t, code, request.AccessCode())
		assert.Equal(t, "<ticket> reqID=1, ####, scopes=[^*]", request.String())
	})

	t.Run("should report errors correctly", func(t *testing.T) {
		request, err := NewTicket(protocol.FirstRequestID, code, nil)
		require.NoError(t, err)
		assert.Equal(t, protocol.Ticket, request.OpCode())
		assert.Equal(t, protocol.FirstRequestID, request.ForRequestID())
		assert.Equal(t, code, request.AccessCode())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		request, err := NewTicket(protocol.FirstRequestID, code, scopes)
		require.NoError(t, err)
		bytes, err := request.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		assert.NoError(t, err)
		assert.Equal(t, request, out)
	})

	t.Run("should report illegal address scopes", func(t *testing.T) {
		request, err := NewTicket(protocol.FirstRequestID, code, nil)
		require.NoError(t, err)
		bytes, err := request.Marshal(protocol.Current)
		require.NoError(t, err)
		bytes = append(bytes, []byte("n@1\000x")...)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Ticket).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report ticket id errors", func(t *testing.T) {
		_, err := NewTicket(protocol.NoRequestID, code, nil)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := createBlank(protocol.Ticket).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Ticket).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Ticket).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Ticket).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(ticketCodeOffset, 0),
			createBlank(protocol.Ticket).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully (no scopes)", func(t *testing.T) {
		broken := make([]byte, ticketCodeOffset)
		injectHeader(broken, protocol.Current, protocol.Ticket)
		broken = append(broken, "code"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Ticket).Unmarshal(protocol.Current, broken))
	})
}
