package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
)

const (
	validateAckIDOffset = headerLength
	validateTokenOffset = validateAckIDOffset + 4
)

// Bind tells a host to use credentials for an token
type Validate interface {
	protocol.Ackable
	Token() OpaqueBytes // remove any doubt around ackID collision
}

type validateMessage struct {
	id    protocol.AckID
	token OpaqueBytes
}

// NewValidate creates a Bind
func NewValidate(id protocol.AckID, token OpaqueBytes) (Validate, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidAckID
	}
	return &validateMessage{id, token}, nil
}

func (*validateMessage) OpCode() protocol.OpCode { return protocol.Validate }
func (v *validateMessage) AckID() protocol.AckID { return v.id }
func (v *validateMessage) Token() OpaqueBytes    { return v.token }

func (v *validateMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, validateTokenOffset+len(v.token))
	injectHeader(bytes, ver, v.OpCode())
	binary.BigEndian.PutUint16(bytes[validateAckIDOffset:], uint16(v.id))
	copy(bytes[validateTokenOffset:], v.token)

	return bytes, nil
}

func (v *validateMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < validateTokenOffset {
		return protocol.ErrInvalidLength(validateTokenOffset, total)
	}
	id := protocol.AckID(binary.BigEndian.Uint16(bytes[validateAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}
	v.id = id
	v.token = OpaqueBytes(bytes[validateTokenOffset:])
	return nil
}

func (v *validateMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, ####", v.OpCode(), v.AckID())
}
