package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestValidate(t *testing.T) {
	token := OpaqueBytes("NotATicket")

	t.Run("should report values correctly", func(t *testing.T) {
		validate, err := NewValidate(protocol.FirstAckID, token)
		require.NoError(t, err)
		assert.Equal(t, protocol.Validate, validate.OpCode())
		assert.Equal(t, protocol.FirstAckID, validate.AckID())
		assert.Equal(t, token, validate.Token())
		assert.Equal(t, "<validate> reqID=1, ####", validate.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		validate, err := NewValidate(protocol.FirstAckID, token)
		require.NoError(t, err)
		bytes, err := validate.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, validate, out)
	})

	t.Run("should report ack id errors", func(t *testing.T) {
		_, err := NewValidate(protocol.NoAckID, token)
		assert.Equal(t, protocol.ErrInvalidAckID, err)
		bytes, err := (&validateMessage{protocol.NoAckID, token}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidAckID, createBlank(protocol.Validate).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Validate).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Validate).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(validateTokenOffset, 0),
			createBlank(protocol.Validate).Unmarshal(protocol.Current, []byte{}))
	})
}
