package protocol

import "strconv"

// Each message header includes a single byte opcode to identity the handler
// that should be processing the payload.
type OpCode byte

const (
	Invalid OpCode = 0 // message could not be parsed

	peerFlag    OpCode = 1 << 5 // broker to broker
	clientFlag  OpCode = 1 << 6 // host to broker
	serviceFlag OpCode = 1 << 7 // broker to host

	Connect  OpCode = peerFlag | 0 // establish initial handshake between nodes
	Forward  OpCode = peerFlag | 1 // tunnel host status/scopes across nodes
	Prepare  OpCode = peerFlag | 2 // begin paxos single-decree
	Promise  OpCode = peerFlag | 3
	Accept   OpCode = peerFlag | 4
	Accepted OpCode = peerFlag | 5
	Ready    OpCode = peerFlag | 6 // mark connection initialization complete; response to connect
	Rejected OpCode = peerFlag | 7 // mark connection incompatible; response to connect

	AuthHost   OpCode = clientFlag | 0 // initial handshake including identity assertion
	Status     OpCode = clientFlag | 1 // includes load and availability
	Ticket     OpCode = clientFlag | 2 // success response for reserve
	Scopes     OpCode = clientFlag | 3 // add or remove supported scopes
	Failure    OpCode = clientFlag | 4 // failure response to reserve, allocate, or detatch
	Allocation OpCode = clientFlag | 5 // success response for allocate
	Validate   OpCode = clientFlag | 6 // attempt to validate a client; returns credentials
	Detached   OpCode = clientFlag | 7 // success response for detatch

	Ack      OpCode = serviceFlag | 1 // authhost/scopes success
	Error    OpCode = serviceFlag | 2 // authhost/scopes/validate error
	Reserve  OpCode = serviceFlag | 3 // attempt to hold a connection slot
	Allocate OpCode = serviceFlag | 4 // attempt to spin up an address on a source
	Bind     OpCode = serviceFlag | 5 // validate auccess
	Detach   OpCode = serviceFlag | 6 // attmept to spin down an address on a source
)

var (
	codeStrings = map[OpCode]string{
		Accept:     "accept",
		Accepted:   "accepted",
		Ack:        "ack",
		Allocate:   "allocate",
		Allocation: "allocation",
		AuthHost:   "auth_host",
		Bind:       "bind",
		Connect:    "connect",
		Detach:     "detach",
		Detached:   "detached",
		Error:      "error",
		Failure:    "failure",
		Forward:    "forward",
		Prepare:    "prepare",
		Promise:    "promise",
		Ready:      "ready",
		Rejected:   "rejected",
		Reserve:    "reserve",
		Scopes:     "scopes",
		Status:     "status",
		Ticket:     "ticket",
		Validate:   "validate",
	}
)

func (o OpCode) IsFromClient() bool  { return (o & clientFlag) != Invalid }
func (o OpCode) IsFromService() bool { return (o & serviceFlag) != Invalid }
func (o OpCode) String() string {
	if name, ok := codeStrings[o]; ok {
		return name
	}
	return "unknown: " + strconv.FormatInt(int64(o), 10)
}
