package protocol

import (
	"encoding/binary"
	"fmt"
)

type SuggestionID interface {
	Next() SuggestionID
	IsValid() bool
	Marshal([]byte) error
	Len() int
	Less(SuggestionID) bool
	Key() SuggestionKey
	String() string
}

type SuggestionKey string

// SuggestionID is a hint used to resolve conflicts in peer-to-peer source
// election following single decrees paxos rules.
type suggestionID struct {
	counter uint32
	from    string
	key     SuggestionKey
}

var NoSuggestion = (*suggestionID)(nil)

func newSuggestion(from string, counter uint32) SuggestionID {
	return &suggestionID{counter, from, SuggestionKey(fmt.Sprintf("%s#%d", from, counter))}
}

func FirstSuggestion(from string) SuggestionID { return newSuggestion(from, 0) }
func UnmarshalSuggestion(bytes []byte) (SuggestionID, error) {
	if len(bytes) < 4 {
		return nil, ErrInvalidLength(4, len(bytes))
	}
	return newSuggestion(string(bytes[4:]), binary.BigEndian.Uint32(bytes)), nil
}

func (s *suggestionID) Next() SuggestionID { return newSuggestion(s.from, s.counter+1) }
func (s *suggestionID) IsValid() bool      { return s != nil }
func (s *suggestionID) Len() int           { return len(s.from) + 4 }
func (s *suggestionID) Key() SuggestionKey { return s.key }
func (s *suggestionID) String() string     { return string(s.key) }
func (s *suggestionID) Less(other SuggestionID) bool {
	if s == nil {
		return other != NoSuggestion
	}
	if cast, ok := other.(*suggestionID); ok && cast != NoSuggestion {
		return s.counter < cast.counter || (s.counter == cast.counter && s.from < cast.from)
	}
	return false
}

func (s *suggestionID) Marshal(bytes []byte) error {
	req := s.Len()
	actual := len(bytes)
	if len(bytes) < req {
		return ErrInvalidLength(req, actual)
	}
	binary.BigEndian.PutUint32(bytes, s.counter)
	copy(bytes[4:], s.from)
	return nil
}
