package auth

import (
	"context"
	"net/http"

	htmid "code.justin.tv/devhub/e2ml/libs/http"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/empty"
)

var (
	// actual key is the address of this var, which is unique in program space
	requestValue = -42
	requestKey   = &requestValue
)

// StoreHandler builds a goji middleware that automatically places a
// Credentials into the context of each request or early outs with an
// unauthorized error.
func StoreExtractor(ext auth.Extractor) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if request, ok := ext.Extract(r.Header); ok {
				ctx := Store(r.Context(), request)
				inner.ServeHTTP(w, r.WithContext(ctx))
			} else {
				htmid.ServeError(w, ext.FailureReason(r.Header))
			}
		})
	}
}

// Store wraps the input context with one containing the specified AuthRequest
func Store(ctx context.Context, auth stream.AuthRequest) context.Context {
	return context.WithValue(ctx, requestKey, auth)
}

// Load retrieves an AuthRequest from a context, defaulting to empty auth if
// no explicit one has been found
func Load(ctx context.Context) stream.AuthRequest {
	if value, ok := ctx.Value(requestKey).(stream.AuthRequest); ok {
		return value
	}
	return empty.NewRequest()
}
