package service

import (
	"net/http"

	"code.justin.tv/infosec/cors"

	goji "goji.io"
	"goji.io/pat"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/service/api"
	"code.justin.tv/devhub/e2ml/libs/discovery/service/auth"
	"code.justin.tv/devhub/e2ml/libs/discovery/service/data"
	htmid "code.justin.tv/devhub/e2ml/libs/http"
	sauth "code.justin.tv/devhub/e2ml/libs/stream/auth"
)

const (
	ConfigStatsFrequent = "stats.frequency.frequent"
	ConfigStatsCommon   = "stats.frequency.common"
	ConfigStatsRare     = "stats.frequency.rare"
)

var (
	corsPolicy = cors.Policy{
		AllowedOrigins: cors.Origins("*"),
		AllowMethods:   cors.Methods("GET", "HEAD"),
		AllowHeaders:   cors.Headers("Accept", "Authorization", "Client-ID", "Content-Type"),
	}
)

// BuildServer combines all parts that make up smart to create functioning handlers
func BuildServer(ext sauth.Extractor, broker discovery.Broker) *goji.Mux {
	server := goji.NewMux()
	server.Use(corsPolicy.MustMiddleware)
	server.Use(cors.BlockOnOptions)
	server.Handle(pat.Get("/health"), htmid.NewAPIHandler("health_check", htmid.Rare, api.HealthCheck))
	BuildV1(server, ext, broker)

	return server
}

func BuildV1(mux *goji.Mux, ext sauth.Extractor, broker discovery.Broker) {
	version1 := goji.SubMux()
	mux.Handle(pat.New(protocol.HttpVersion1Pattern), version1)
	version1.Use(auth.StoreExtractor(ext))
	version1.Use(data.StoreBroker(broker))

	version1.Handle(Pattern(protocol.GetHostMethod, protocol.GetHostPath),
		htmid.NewAPIHandler("get_host", htmid.Frequent, api.GetHost))
}

func Pattern(method, path string) *pat.Pattern {
	if method == http.MethodGet {
		return pat.NewWithMethods(path, method, http.MethodHead)
	}
	return pat.NewWithMethods(path, method)
}
