package errors

// Details defines the public interface for expressing error parameters
type Details map[string]interface{}

// String extracts a String from details
func (d Details) String(key string) string {
	if d == nil {
		return ""
	}
	if cast, ok := d[key].(string); ok {
		return cast
	}
	return ""
}

// Int extracts an int from details
func (d Details) Int64(key string) int64 {
	if d == nil {
		return 0
	}
	switch cast := d[key].(type) {
	case int:
		return int64(cast)
	case int16:
		return int64(cast)
	case int32:
		return int64(cast)
	case int64:
		return cast
	case float32:
		return int64(cast)
	case float64:
		return int64(cast)
	case uint16:
		return int64(cast)
	case uint32:
		return int64(cast)
	case uint64:
		return int64(cast)
	}
	return 0
}

// DetailsSource returns an arbitrary map of parameters
type DetailsSource interface {
	Details() Details
}

// DetailsError is an error with Details
type DetailsError interface {
	DetailsSource
	error
}

// DetailsAdaptable can return a copy as an DetailsError
type DetailsAdaptable interface {
	WithDetails(details Details) DetailsError
}

// WithDetails attaches or replaces the error code on an existing error
func WithDetails(e error, details Details) DetailsError {
	if cast, ok := e.(DetailsAdaptable); ok {
		return cast.WithDetails(details)
	}
	return extract(e).WithDetails(details)
}

// GetDetails extracts the details from an error or returns an empty set
func GetDetails(e error) Details {
	if cast, ok := e.(DetailsSource); ok {
		return cast.Details()
	}
	return Details{}
}
