package errors

import (
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestGetDetails(t *testing.T) {
	t.Run("Should allow typed string extractions", func(t *testing.T) {
		details := Details{"key": "value", "key2": 012345}
		assert.Equal(t, "value", details.String("key"))
		assert.Equal(t, "", details.String("key2"))
		assert.Equal(t, "", Details(nil).String("key2"))
	})

	t.Run("Should allow typed int64 extractions", func(t *testing.T) {
		details := Details{
			"key":    -123,
			"string": "value",
			"key16":  int16(16),
			"key32":  int32(32),
			"key64":  int64(64),
			"keyu16": uint16(16),
			"keyu32": uint32(32),
			"keyu64": uint64(64),
			"keyf32": float32(32),
			"keyf64": float64(64),
		}
		assert.Equal(t, int64(-123), details.Int64("key"))
		assert.Equal(t, int64(16), details.Int64("key16"))
		assert.Equal(t, int64(32), details.Int64("key32"))
		assert.Equal(t, int64(64), details.Int64("key64"))
		assert.Equal(t, int64(16), details.Int64("keyu16"))
		assert.Equal(t, int64(32), details.Int64("keyu32"))
		assert.Equal(t, int64(64), details.Int64("keyu64"))
		assert.Equal(t, int64(0), details.Int64("string"))
		assert.Equal(t, int64(32), details.Int64("keyf32"))
		assert.Equal(t, int64(64), details.Int64("keyf64"))
		assert.Equal(t, int64(0), details.Int64("missing"))
		assert.Equal(t, int64(0), Details(nil).Int64("missing"))
	})

	t.Run("Should return empty by default", func(t *testing.T) {
		err := errors.New("raw")
		assert.Equal(t, Details{}, GetDetails(err))
	})

	t.Run("Should work on a manually decorated error", func(t *testing.T) {
		err := errors.New("raw")
		err = WithDetails(err, Details{"foo": "bar"})
		assert.Equal(t, Details{"foo": "bar"}, GetDetails(err))
	})

	t.Run("Should work on a built error", func(t *testing.T) {
		var err error = NewBuilder("message").WithDetails(Details{"foo": "baz"}).Build()
		err = WithDetails(err, Details{"foo": "bar"})
		assert.Equal(t, Details{"foo": "bar"}, GetDetails(err))
	})
}
