package errors

import (
	"encoding/json"
	"fmt"
)

// A Dictionary allows errors to be marshaled by error code with optional details and
// properly reconstructed on the other end of the wire, allowing the most compact
// representation possible.
type Dictionary interface {
	Get(code string, details Details) error
	Marshal(error) ([]byte, error)
	Unmarshal([]byte, *error) error
}

type dictionary map[string]func(Details) error

func (d dictionary) Get(code string, details Details) error {
	if callback, matched := d.find(&code); matched {
		return callback(details)
	}
	if details == nil {
		return NewBuilder(code).WithErrorCode(code).Build()
	}
	msg := code
	for k, v := range details {
		msg += fmt.Sprintf("|%s:%v", k, v)
	}
	return NewBuilder(msg).WithErrorCode(code).WithDetails(details).Build()
}

func (d dictionary) Marshal(err error) ([]byte, error) {
	converted := extract(err)
	// if the builder has this error code, remove message and status to make it more compact
	if _, matched := d.find(converted.ErrorCode); matched {
		converted.Message = ""
		converted.HTTPStatus = nil
	}
	return json.Marshal(converted)
}

func (d dictionary) Unmarshal(bytes []byte, result *error) error {
	converted := &errorImpl{}
	err := json.Unmarshal(bytes, converted)
	if err != nil {
		return err
	}
	if converted.ErrorCode == nil {
		*result = converted.wrap()
		return nil
	}
	details := Details(nil)
	if converted.Details != nil {
		details = *converted.Details
	}
	*result = d.Get(*converted.ErrorCode, details)
	return nil
}

func (d dictionary) find(code *string) (func(Details) error, bool) {
	if d != nil && code != nil {
		val, found := d[*code]
		return val, found
	}
	return nil, false
}

func (d dictionary) copy() dictionary {
	result := make(dictionary)
	if d != nil {
		for k, v := range d {
			result[k] = v
		}
	}
	return result
}
