package errors

import (
	"encoding/json"
	"io"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDictionary(t *testing.T) {
	t.Run("should build sensible defaults for unknown errors without details", func(t *testing.T) {
		expected := NewBuilder("test").WithErrorCode("test").Build()
		dict := make(dictionary)
		assert.Equal(t, expected, dict.Get("test", nil))
	})

	t.Run("should build sensible defaults for unknown errors with details", func(t *testing.T) {
		details := Details{"key": "value"}
		expected := NewBuilder("test|key:value").WithErrorCode("test").WithDetails(details).Build()
		dict := make(dictionary)
		assert.Equal(t, expected, dict.Get("test", details))
	})

	t.Run("should return known errors", func(t *testing.T) {
		expected := NewBuilder("test").Build().WithErrorCode("code")
		expected2 := NewBuilder("test").Build().WithErrorCode("code2")
		dict := NewDictionaryBuilder().Include(expected, expected2).Build()
		assert.Equal(t, expected, dict.Get(expected.ErrorCode(), nil))
		assert.Equal(t, expected2, dict.Get(expected2.ErrorCode(), nil))
	})

	t.Run("should allow composition", func(t *testing.T) {
		expected := NewBuilder("test").Build().WithErrorCode("code")
		expected2 := NewBuilder("test").Build().WithErrorCode("code2")
		dict1 := NewDictionaryBuilder().Include(expected).Build()
		dict2 := NewDictionaryBuilder().Include(expected2).Build()
		dict := NewDictionaryBuilder().IncludeAll(dict1).IncludeAll(dict2).Build()
		assert.Equal(t, expected, dict.Get(expected.ErrorCode(), nil))
		assert.Equal(t, expected2, dict.Get(expected2.ErrorCode(), nil))
	})

	t.Run("should construct registered types", func(t *testing.T) {
		code := (&ErrExtensionNotFound{}).ErrorCode()
		dict := NewDictionaryBuilder().Map(code, func(details Details) error {
			id := ""
			if cast, ok := details["id"].(string); ok {
				id = cast
			}
			return NewErrExtensionNotFound(id)
		}).Build()
		assert.IsType(t, &ErrExtensionNotFound{}, dict.Get(code, nil))
	})

	t.Run("should marshal unregisgtered types", func(t *testing.T) {
		value := NewErrExtensionNotFound("value")
		code := (&ErrExtensionNotFound{}).ErrorCode()
		dict := NewDictionaryBuilder().Map(code, func(details Details) error {
			id := ""
			if cast, ok := details["id"].(string); ok {
				id = cast
			}
			return NewErrExtensionNotFound(id)
		}).Build()
		bytes, err := dict.Marshal(value)
		require.NoError(t, err)
		assert.Equal(t, []byte(`{"error_code":"extension_not_found","details":{"id":"value"}}`), bytes)
		var result error
		require.NoError(t, dict.Unmarshal(bytes, &result))
		assert.Equal(t, value, result)
	})

	t.Run("should marshal registered types", func(t *testing.T) {
		value := NewErrExtensionNotFound("value")
		code := (&ErrExtensionNotFound{}).ErrorCode()
		dict := NewDictionaryBuilder().Map(code, func(details Details) error {
			id := ""
			if cast, ok := details["id"].(string); ok {
				id = cast
			}
			return NewErrExtensionNotFound(id)
		}).Build()
		bytes, err := dict.Marshal(value)
		require.NoError(t, err)
		assert.Equal(t, []byte(`{"error_code":"extension_not_found","details":{"id":"value"}}`), bytes)
		var result error
		require.NoError(t, dict.Unmarshal(bytes, &result))
		assert.Equal(t, value, result)
	})

	t.Run("should forward errors without codes", func(t *testing.T) {
		dict := NewDictionaryBuilder().Build()
		bytes, err := dict.Marshal(io.EOF)
		require.NoError(t, err)
		assert.Equal(t, []byte(`{"message":"EOF"}`), bytes)
		var result error
		require.NoError(t, dict.Unmarshal(bytes, &result))
		assert.EqualError(t, io.EOF, "EOF")
	})

	t.Run("should forward marshal errors", func(t *testing.T) {
		src := NewBuilder("x").WithDetails(Details{"x": &badMarshal{}}).Build()
		dict := NewDictionaryBuilder().Build()
		result, err := dict.Marshal(src)
		assert.Nil(t, result)
		assert.IsType(t, &json.MarshalerError{}, err)
	})

	t.Run("should forward unmarshal errors", func(t *testing.T) {
		dict := NewDictionaryBuilder().Build()
		var result error
		err := dict.Unmarshal([]byte("{{}"), &result)
		assert.Nil(t, result)
		assert.IsType(t, &json.SyntaxError{}, err)
	})
}
