package errors

import (
	"fmt"
)

type ErrExtensionNotFound struct{ id string }

func NewErrExtensionNotFound(id string) error    { return &ErrExtensionNotFound{id} }
func (e *ErrExtensionNotFound) Error() string    { return "Extension \"" + e.id + "\" could not be found" }
func (*ErrExtensionNotFound) ErrorCode() string  { return "extension_not_found" }
func (*ErrExtensionNotFound) HTTPStatus() int    { return 404 }
func (e *ErrExtensionNotFound) Details() Details { return Details{"id": e.id} }

func ExampleErrorBuilder() {
	var unauthorized = NewBuilder("Unauthorized").
		WithErrorCode("unauthorized").
		WithHTTPStatus(401).
		Build()
	value, _ := MarshalAny(unauthorized)
	fmt.Println(string(value))

	//Output: {"message":"Unauthorized","error_code":"unauthorized","status":401}
}

func ExampleStructuredError() {
	var missing = NewErrExtensionNotFound("myextensionid")
	value, _ := MarshalAny(missing)
	fmt.Println(string(value))

	//Output: {"message":"Extension \"myextensionid\" could not be found","error_code":"extension_not_found","details":{"id":"myextensionid"},"status":404}
}

func ExampleDictionary() {
	var missing = NewErrExtensionNotFound("myextensionid")
	dict := NewDictionaryBuilder().Map(GetErrorCode(missing), func(details Details) error {
		return NewErrExtensionNotFound(details.String("id"))
	}).Build()
	value, _ := dict.Marshal(missing)
	fmt.Println(string(value))
	var result error
	_ = dict.Unmarshal(value, &result)
	fmt.Println(result)

	//Output: {"error_code":"extension_not_found","details":{"id":"myextensionid"}}
	// Extension "myextensionid" could not be found
}
