package errors

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestWrap(t *testing.T) {
	details := Details{"x": "y"}
	status := 1
	code := "code"
	msg := "message"

	t.Run("for no decoration", func(t *testing.T) {
		e := (&errorImpl{Message: msg}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Empty(t, GetDetails(e))
		assert.Empty(t, GetHTTPStatusOrDefault(e, 0))
		assert.Empty(t, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for error code only", func(t *testing.T) {
		e := (&errorImpl{Message: msg, ErrorCode: &code}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Empty(t, GetDetails(e))
		assert.Empty(t, GetHTTPStatusOrDefault(e, 0))
		assert.Equal(t, code, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for http status only", func(t *testing.T) {
		e := (&errorImpl{Message: msg, HTTPStatus: &status}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Empty(t, GetDetails(e))
		assert.Equal(t, status, GetHTTPStatusOrDefault(e, 0))
		assert.Empty(t, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for error code + http status", func(t *testing.T) {
		e := (&errorImpl{Message: msg, ErrorCode: &code, HTTPStatus: &status}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Empty(t, GetDetails(e))
		assert.Equal(t, status, GetHTTPStatusOrDefault(e, 0))
		assert.Equal(t, code, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for details only", func(t *testing.T) {
		e := (&errorImpl{Message: msg, Details: &details}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Equal(t, details, GetDetails(e))
		assert.Empty(t, GetHTTPStatusOrDefault(e, 0))
		assert.Empty(t, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for details + error code", func(t *testing.T) {
		e := (&errorImpl{Message: msg, Details: &details, ErrorCode: &code}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Equal(t, details, GetDetails(e))
		assert.Empty(t, GetHTTPStatusOrDefault(e, 0))
		assert.Equal(t, code, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for http status + error code", func(t *testing.T) {
		e := (&errorImpl{Message: msg, HTTPStatus: &status, ErrorCode: &code}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Empty(t, GetDetails(e))
		assert.Equal(t, status, GetHTTPStatusOrDefault(e, 0))
		assert.Equal(t, code, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for http status + details", func(t *testing.T) {
		e := (&errorImpl{Message: msg, HTTPStatus: &status, Details: &details}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Equal(t, details, GetDetails(e))
		assert.Equal(t, status, GetHTTPStatusOrDefault(e, 0))
		assert.Empty(t, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})

	t.Run("for error code + http status + details", func(t *testing.T) {
		e := (&errorImpl{
			Message:    msg,
			ErrorCode:  &code,
			HTTPStatus: &status,
			Details:    &details,
		}).wrap()
		assert.Equal(t, msg, e.Error())
		assert.Equal(t, details, GetDetails(e))
		assert.Equal(t, status, GetHTTPStatusOrDefault(e, 0))
		assert.Equal(t, code, GetErrorCode(e))
		assert.IsType(t, &errorImpl{}, e.unwrap())
	})
}
