package http

import (
	"context"
	"encoding/json"
	"io"
	"net/http"
	"strconv"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"code.justin.tv/devhub/e2ml/libs/http/params"
)

// Frequency is a hint to an Instrumentor about how often an Event is expected to occur
type Frequency string

const (
	// Rare -- should be used for things that aren't part of common viewer/broadcaster use
	Rare Frequency = "rare"
	// Common -- should be used for things that viewers/broadcasters use periodically
	Common Frequency = "common"
	// Frequent -- should be used for things that will be repeated on a per-viewer/broadcaster basis
	Frequent Frequency = "frequent"
)

// APICall defines a generic interface for functions to be plugged into an HTTP
// service.
type APICall func(context.Context) (interface{}, error)

// APIHandler wraps an APICall for use with a Mux
type APIHandler struct {
	call APICall
	name string
	freq Frequency
}

// NewAPIHandler creates an adapter around an API call and registers a hystrix
// circuit for the handler.
func NewAPIHandler(name string, freq Frequency, call APICall) http.Handler {
	return &APIHandler{call, name, freq}
}

func (h *APIHandler) Name() string         { return h.name }
func (h *APIHandler) Frequency() Frequency { return h.freq }
func (h *APIHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	defer tryClose(r.Body)
	ctx := params.ExtractFromHTTP(r.Context(), r)
	out, err := h.call(ctx)
	if err != nil {
		code := errors.GetHTTPStatusOrDefault(err, 500)
		// Serve 2XX responses (201, 206, etc.)
		if code >= 200 && code < 300 {
			w.WriteHeader(code)
			ServeJSON(w, out)
		} else {
			ServeError(w, err)
		}
		return
	}

	ServeJSON(w, out)
}

func ServeJSON(w http.ResponseWriter, out interface{}) {
	if out == nil {
		w.WriteHeader(http.StatusNoContent)
		return
	}
	content, err := json.MarshalIndent(out, "", "  ")
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json")
	w.Write(content)
}

func ServeError(w http.ResponseWriter, err error) {
	w.WriteHeader(errors.GetHTTPStatusOrDefault(err, 500))

	body, errm := errors.MarshalAny(err)
	if errm != nil {
		return // ignored
	}

	w.Header().Set("Content-Length", strconv.Itoa(len(body)))
	w.Header().Set("Content-Type", "application/json")
	w.Write(body)
}

func tryClose(body io.Closer) {
	if body != nil {
		_ = body.Close()
	}
}
