package http

import (
	"io"
	"io/ioutil"
	"net/http"
	"strings"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type brokenReadCloser struct{}

func (*brokenReadCloser) Error() string                      { return "expected" }
func (b *brokenReadCloser) Read(p []byte) (n int, err error) { return 0, b }
func (b *brokenReadCloser) Close() error                     { return b }

func errForResponse(err error) io.ReadCloser {
	bytes, _ := errors.MarshalAny(err)
	return ioutil.NopCloser(strings.NewReader(string(bytes)))
}

func TestGenericErrorForHTTPStatus(t *testing.T) {
	t.Run("should not return an error for non-error codes", func(t *testing.T) {
		err, isError := GenericErrorForHTTPStatus(100)
		assert.False(t, isError)
		assert.Nil(t, err)
	})

	t.Run("should return standard errors for known codes", func(t *testing.T) {
		err, isError := GenericErrorForHTTPStatus(http.StatusNotFound)
		assert.True(t, isError)
		assert.Equal(t, err, ErrNotFound)
		assert.Equal(t, err.Error(), "404: Not Found")
		assert.Equal(t, err.HTTPStatus(), http.StatusNotFound)
		assert.Equal(t, err.ErrorCode(), "http_not_found")
	})

	t.Run("should return a templated error for unknown codes", func(t *testing.T) {
		err, isError := GenericErrorForHTTPStatus(9001)
		assert.True(t, isError)
		require.NotNil(t, err)
		assert.Equal(t, err.Error(), "HTTP Error: 9001")
		assert.Equal(t, err.HTTPStatus(), 9001)
		assert.Equal(t, err.ErrorCode(), "http_9001")
	})
}

func TestExtractFromHTTPResponse(t *testing.T) {
	t.Run("should return no error for successful response", func(t *testing.T) {
		err, isError := ExtractErrorFromResponse(&http.Response{StatusCode: 204})
		assert.False(t, isError)
		assert.Nil(t, err)
	})

	t.Run("should return custom client error intact", func(t *testing.T) {
		src := errors.NewBuilder("Expected").WithHTTPStatus(400).WithErrorCode("expected").Build()

		err, isError := ExtractErrorFromResponse(&http.Response{
			StatusCode: 400,
			Body:       errForResponse(src),
		})
		assert.True(t, isError)
		assert.Equal(t, err.Error(), "Expected")
	})

	t.Run("should return standard error if custom can't be read", func(t *testing.T) {
		err, isError := ExtractErrorFromResponse(&http.Response{
			StatusCode: 400,
			Body:       &brokenReadCloser{},
		})
		assert.True(t, isError)
		assert.Equal(t, err, ErrBadRequest)
	})

	t.Run("should return standrd error for unknown service issues", func(t *testing.T) {
		src := errors.NewBuilder("Custom").WithHTTPStatus(400).Build()
		err, isError := ExtractErrorFromResponse(&http.Response{
			StatusCode: 503,
			Body:       errForResponse(src),
		})
		assert.True(t, isError)
		assert.Equal(t, ErrServiceUnavailable, err) // uses status code, not body code
	})
}
