package http

import (
	"crypto/tls"
	"errors"
	"net"
	"net/http"
	"sync/atomic"
	"time"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/session"
)

type service struct {
	port        int
	handler     http.Handler
	settings    *Settings
	hasShutdown int32
	isRunning   int32
}

type ServiceFactory func(http.Handler) (session.Service, error)

func NewService(port int, handler http.Handler, settings *Settings) (session.Service, error) {
	return &service{port: port, handler: handler, settings: settings}, nil
}

func NewServiceFactory(port int, settings *Settings) ServiceFactory {
	return func(handler http.Handler) (session.Service, error) {
		return &service{port: port, handler: handler, settings: settings}, nil
	}
}

func (s *service) Start() error {
	atomic.StoreInt32(&s.isRunning, 1)
	ln, err := s.listen()
	if err != nil {
		return err
	}
	s.settings.Lifecycle.RunUntilComplete(func() { s.run(ln) })
	return nil
}

func (s *service) Stop() {
	atomic.StoreInt32(&s.isRunning, 0)
	s.settings.Lifecycle.ExecuteHook(s)
}

func (s *service) WaitForDrainingConnections(until time.Time) {
	s.settings.Lifecycle.WaitForCompletion(until)
}

func (s *service) IsRunning() bool {
	return atomic.LoadInt32(&s.isRunning) != 0
}

func (s *service) HasShutdown() bool {
	return atomic.LoadInt32(&s.hasShutdown) != 0
}

func (s *service) Shutdown() error {
	s.Stop()
	atomic.AddInt32(&s.hasShutdown, 1)
	return s.settings.Lifecycle.ExecuteAll()
}

func (s *service) run(ln net.Listener) {
	defer s.settings.Logger(logging.Debug, "Leaving accept loop")
	for {
		err := http.Serve(ln, s.handler)
		if err == nil {
			continue
		}
		if isClosedError(err) {
			return
		}
		s.settings.Logger(logging.Debug, "Unable to accept connection:", err)
		if ln, err = s.listen(); err != nil {
			s.settings.Logger(logging.Warning, "Unable to listen: ", err)
			time.Sleep(time.Second)
		}
	}
}

func (s *service) listen() (net.Listener, error) {
	tc, err := net.ListenTCP("tcp", &net.TCPAddr{Port: s.port})
	if err != nil {
		return nil, err
	}

	ln := withKeepalive(tc)
	if s.settings.Certs != nil {
		ln = tls.NewListener(ln, s.settings.Certs)
	}
	s.settings.Lifecycle.RegisterHook(s, ln.Close)
	s.settings.Logger(logging.Info, "Listening on port", s.port, "...")
	if s.HasShutdown() || !s.IsRunning() {
		s.settings.Lifecycle.ExecuteHook(s) // catch race condition - shutdown during listen request
	}
	return ln, nil
}

func isClosedError(err error) bool {
	cast := new(net.OpError)
	return errors.As(err, &cast) && cast.Err.Error() == "use of closed network connection"
}
