package logged

import (
	"fmt"
	"sort"
	"strings"
	"sync"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
)

type tracker struct {
	logger      logging.Function
	level       logging.Level
	namespace   string
	elements    metricList
	counts      map[string]*count
	timings     map[string]*timing
	gauges      map[string]*gauge
	aggregators map[string]*aggregator
	mutex       sync.RWMutex
}

// NewTracker Creates a new tracker that writes values to log
func NewTracker(namespace string, level logging.Level, logger logging.Function) (metrics.Tracker, error) {
	return &tracker{
		logger:      logger,
		level:       level,
		namespace:   namespace,
		counts:      make(map[string]*count),
		timings:     make(map[string]*timing),
		gauges:      make(map[string]*gauge),
		aggregators: make(map[string]*aggregator),
	}, nil
}

func (*tracker) String() string { return "{logged metrics tracker}" }

func (t *tracker) Tick() {
	t.mutex.RLock()
	eval := make(metricList, len(t.elements))
	copy(eval, t.elements)
	t.mutex.RUnlock()
	for _, m := range eval {
		if id, value, ok := m.tick(); ok {
			t.log(id, value)
		}
	}
}

func (t *tracker) Close() error {
	t.Tick()
	return nil
}

func (t *tracker) Aggregator(name string, tags []string) metrics.Aggregator {
	key := createKey(name, tags)
	t.mutex.RLock()
	a, ok := t.aggregators[key]
	t.mutex.RUnlock()
	if ok {
		return a
	}
	t.mutex.Lock()
	if a, ok = t.aggregators[key]; !ok {
		a = &aggregator{key, t.buildID("aggre", name, tags), 0, 0}
		t.aggregators[key] = a
		t.inject(a)
	}
	t.mutex.Unlock()
	return a
}

func (t *tracker) Count(name string, tags []string) metrics.Count {
	key := createKey(name, tags)
	t.mutex.RLock()
	c, ok := t.counts[key]
	t.mutex.RUnlock()
	if ok {
		return c
	}
	t.mutex.Lock()
	if c, ok = t.counts[key]; !ok {
		c = &count{key, t.buildID("count", name, tags), 0}
		t.counts[key] = c
		t.inject(c)
	}
	t.mutex.Unlock()
	return c
}

func (t *tracker) Timing(name string, tags []string) metrics.Timing {
	key := createKey(name, tags)
	t.mutex.RLock()
	tm, ok := t.timings[key]
	t.mutex.RUnlock()
	if ok {
		return tm
	}
	t.mutex.Lock()
	if tm, ok = t.timings[key]; !ok {
		tm = &timing{t.buildID("timing", name, tags), t}
		t.timings[key] = tm
	}
	t.mutex.Unlock()
	return tm
}

func (t *tracker) Gauge(name string, tags []string) metrics.Gauge {
	key := createKey(name, tags)
	t.mutex.RLock()
	g, ok := t.gauges[key]
	t.mutex.RUnlock()
	if ok {
		return g
	}
	t.mutex.Lock()
	if g, ok = t.gauges[key]; !ok {
		g = &gauge{key, t.buildID("gauge", name, tags), 0, 0}
		t.gauges[key] = g
		t.inject(g)
	}
	t.mutex.Unlock()
	return g
}

func createKey(name string, tags []string) string {
	if len(tags) == 0 {
		return name
	}
	sorted := make([]string, len(tags))
	copy(sorted, tags)
	sort.Strings(sorted)
	return fmt.Sprintf("%s|%v", name, strings.Join(sorted, "|"))
}

func (t *tracker) buildID(typ, name string, tags []string) string {
	tagMap := make(map[string]struct{})
	for _, tag := range tags {
		tagMap[tag] = struct{}{}
	}
	sorted := make([]string, 0, len(tagMap))
	for tag := range tagMap { // strip unknowns for readability
		if strings.HasSuffix(tag, ":unknown") {
			continue
		}
		sorted = append(sorted, tag)
	}
	if len(sorted) == 0 {
		return fmt.Sprintf("{%s}[%s%s]", typ, t.namespace, name)
	}
	sort.Strings(sorted)
	return fmt.Sprintf("{%s}[%s%s %s]", typ, t.namespace, name, strings.Join(sorted, ", "))
}

func (t *tracker) log(id string, value float64) {
	t.logger(t.level, id, value)
}

func (t *tracker) inject(m metric) {
	index := sort.Search(len(t.elements), func(i int) bool { return t.elements[i].key() >= m.key() })
	t.elements = append(t.elements, m)
	copy(t.elements[index+1:], t.elements[index:])
	t.elements[index] = m
}
