package twitchtelemetry

import (
	"fmt"
	"math"
	"sort"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
)

type tracker struct {
	reporter telemetry.SampleReporter

	gauges      map[string]*gauge
	aggregators map[string]*aggregator
	mutex       sync.RWMutex
}

// NewTracker returns a metrics.Tracker implementation using TwitchTelemetry with CloudWatch.
// Gauges and aggregators are implemented by reporting a count on every tick, wihch should
// be display in graphs with the "Average" statistic in CloudWatch.
// Count and Timing are implemented by calling Report directly on the telemetry.SampleReporter.
func NewTracker(tPid identifier.ProcessIdentifier, logger logging.Function) metrics.Tracker {
	builder, observer := SampleBuilderAndObserver(tPid, 30*time.Second)
	tLogger := &Logger{Level: logging.Info, Logger: logger}

	reporter := telemetry.SampleReporter{
		SampleBuilder:  builder,
		SampleObserver: observer,
		Logger:         tLogger,
	}

	return &tracker{
		reporter:    reporter,
		gauges:      make(map[string]*gauge),
		aggregators: make(map[string]*aggregator),
	}
}

func (t *tracker) Report(metricName string, value float64, units string) {
	t.reporter.Report(metricName, value, units)
}

func (t *tracker) ReportDurationSample(metricName string, duration time.Duration) {
	t.reporter.ReportDurationSample(metricName, duration)
}

func (*tracker) String() string {
	return "{TwitchTelemetry metrics tracker}"
}

func (t *tracker) Tick() {
	t.mutex.RLock()
	defer t.mutex.RUnlock()
	for _, g := range t.gauges {
		g.tick()
	}
	for _, a := range t.aggregators {
		a.tick()
	}
}

func (t *tracker) Close() error {
	t.Tick()
	t.reporter.SampleObserver.Flush()
	t.reporter.SampleObserver.Stop()
	return nil
}

// Gauge sets a value in memory and reports periodically.
// Use "Average" statistic in CloudWatch to see gauge values (do not use Sum).
func (t *tracker) Gauge(name string, tags []string) metrics.Gauge {
	key := createKey("gauge", name, tags)
	t.mutex.RLock()
	g, ok := t.gauges[key]
	t.mutex.RUnlock()
	if ok {
		return g
	}
	t.mutex.Lock()
	g, ok = t.gauges[key]
	if !ok {
		g = &gauge{t, key, 0}
		t.gauges[key] = g
	}
	t.mutex.Unlock()
	return g
}

type gauge struct {
	t     *tracker
	key   string
	value uint64
}

func (g *gauge) Set(value float64) {
	atomic.StoreUint64(&g.value, math.Float64bits(value))
}
func (g *gauge) tick() {
	value := math.Float64frombits(atomic.LoadUint64(&g.value))
	g.t.Report(g.key, value, telemetry.UnitCount)
}

// Aggregator accumulates a gauge in memory and reports periodically.
// Use "Average" statistic in CloudWatch to see gauge values (do not use Sum).
func (t *tracker) Aggregator(name string, tags []string) metrics.Aggregator {
	key := createKey("gauge", name, tags)
	t.mutex.RLock()
	a, ok := t.aggregators[key]
	t.mutex.RUnlock()
	if ok {
		return a
	}
	t.mutex.Lock()
	a, ok = t.aggregators[key]
	if !ok {
		a = &aggregator{t, key, 0}
		t.aggregators[key] = a
	}
	t.mutex.Unlock()
	return a
}

type aggregator struct {
	t     *tracker
	key   string
	value uint64
}

func (a *aggregator) Set(value float64) {
	atomic.StoreUint64(&a.value, math.Float64bits(value))
}

func (a *aggregator) Add(value int64) {
	for {
		prev := atomic.LoadUint64(&a.value)
		calc := math.Float64frombits(prev) + float64(value)
		if atomic.CompareAndSwapUint64(&a.value, prev, math.Float64bits(calc)) {
			return
		}
	}
}

func (a *aggregator) tick() {
	value := math.Float64frombits(atomic.LoadUint64(&a.value))
	a.t.Report(a.key, value, telemetry.UnitCount)
}

// Count reports an incremental value for a period of time
func (t *tracker) Count(name string, tags []string) metrics.Count {
	key := createKey("count", name, tags)
	return &count{t, key}
}

type count struct {
	t   *tracker
	key string
}

func (c *count) Add(value int64) {
	c.t.Report(c.key, float64(value), telemetry.UnitCount)
}

// Timing reports a duration sample
func (t *tracker) Timing(name string, tags []string) metrics.Timing {
	key := createKey("timing", name, tags)
	return &timing{t, key}
}

type timing struct {
	t   *tracker
	key string
}

func (t *timing) Sample(value time.Duration) {
	t.t.ReportDurationSample(t.key, value)
}

// createKey converts the Datadog style metic name with tags into a CloudWatch metric name.
// e.g. createKey("gauge", "broker.auth", string[]{"action:validate", "result:ok"}) => "gauge.broker.auth;action:validate,result:ok"
func createKey(prefix, name string, tags []string) string {
	if len(tags) == 0 {
		return fmt.Sprintf("%s.%s", prefix, name)
	}

	sorted := make([]string, len(tags))
	copy(sorted, tags)
	sort.Strings(sorted)
	return fmt.Sprintf("%s.%s;%v", prefix, name, strings.Join(sorted, ","))
}
