package dns

import (
	"net"
	"strings"
	"sync"
	"sync/atomic"
	"time"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/peering"
)

type listMap map[*serverList]struct{}
type nameMap map[string]struct{}

type listSource struct {
	hostname string
	reported unsafe.Pointer
	attached unsafe.Pointer
	close    chan struct{}
	wg       sync.WaitGroup
}

func createSource(hostname string) *listSource {
	r := make(nameMap)
	a := make(listMap)
	return &listSource{
		hostname: hostname,
		reported: unsafe.Pointer(&r),
		attached: unsafe.Pointer(&a),
		close:    make(chan struct{}),
	}
}

func (l *listSource) start(period time.Duration) {
	defer l.wg.Done()
	ticker := time.NewTicker(period)
	for {
		if hosts, err := net.LookupHost(l.hostname); err == nil {
			l.update(hosts)
		}
		select {
		case <-l.close:
			ticker.Stop()
			return
		case <-ticker.C:
		}
	}
}

func (l *listSource) update(addrs []string) {
	reported := make(nameMap)
	for _, val := range addrs {
		if strings.Contains(val, ":") { // Consider IPv4 only
			continue
		}
		reported[val] = struct{}{}
	}
	prev := *(*nameMap)(atomic.SwapPointer(&l.reported, unsafe.Pointer(&reported)))
	lists := l.lists()
	for key := range prev {
		if _, ok := reported[key]; !ok {
			for list := range lists {
				list.onRemoved(key)
			}
		}
	}
	for key := range reported {
		if _, ok := prev[key]; !ok {
			for list := range lists {
				list.onAdded(key)
			}
		}
	}
}

func (l *listSource) names() nameMap { return *(*nameMap)(atomic.LoadPointer(&l.reported)) }
func (l *listSource) lists() listMap { return *(*listMap)(atomic.LoadPointer(&l.attached)) }

func (l *listSource) create(name string) peering.ClosableServerList {
	list := newList(l, name)
	for {
		prev := atomic.LoadPointer(&l.attached)
		lists := *(*listMap)(prev)
		copy := make(listMap)
		for k, v := range lists {
			copy[k] = v
		}
		copy[list] = struct{}{}
		if atomic.CompareAndSwapPointer(&l.attached, prev, unsafe.Pointer(&copy)) {
			break
		}
	}
	// inject copy into prev
	return list
}

func (l *listSource) onClosed(list *serverList) error {
	for {
		prev := atomic.LoadPointer(&l.attached)
		lists := *(*listMap)(prev)
		copy := make(listMap)
		for k, v := range lists {
			copy[k] = v
		}
		delete(copy, list)
		if atomic.CompareAndSwapPointer(&l.attached, prev, unsafe.Pointer(&copy)) {
			if len(copy) == 0 {
				close(l.close)
				l.wg.Wait()
			}
			break
		}
	}
	return nil
}

func (l *listSource) onAdded(listener peering.Listener) {
	for name := range l.names() {
		listener.OnPeerAdded(name)
	}
}
