package dns

import (
	"sync/atomic"
	"time"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/peering"
)

type source interface {
	create(string) peering.ClosableServerList
	onAdded(peering.Listener)
	onClosed(*serverList) error
}

type serverList struct {
	source      source
	local       string
	listPointer unsafe.Pointer
	closed      int32
}

// NewServerList attempts to get a peer listing by DNS A records
func NewServerList(hostname, local string, period time.Duration) peering.ClosableServerList {
	src := createSource(hostname)
	src.wg.Add(1)
	go src.start(period)
	return src.create(local)
}

func newList(src source, name string) *serverList {
	return &serverList{
		source:      src,
		local:       name,
		listPointer: unsafe.Pointer(&[]peering.Listener{}),
	}
}

func (s *serverList) onRemoved(name string) {
	if name == s.local {
		return
	}
	listeners := s.listeners()
	for _, l := range listeners {
		l.OnPeerRemoved(name)
	}
}

func (s *serverList) onAdded(name string) {
	if name == s.local {
		return
	}
	listeners := s.listeners()
	for _, l := range listeners {
		l.OnPeerAdded(name)
	}
}

func (s *serverList) Close() error {
	if atomic.SwapInt32(&s.closed, 1) == 0 {
		return s.source.onClosed(s)
	}
	return nil
}

func (s *serverList) LocalName() string                                { return s.local }
func (s *serverList) WithLocal(name string) peering.ClosableServerList { return s.source.create(name) }
func (s *serverList) AddListener(listener peering.Listener) {
	for {
		prev, listeners := s.removeListenerInternal(listener)
		listeners = append(listeners, listener)
		if atomic.CompareAndSwapPointer(&s.listPointer, prev, unsafe.Pointer(&listeners)) {
			s.source.onAdded(listener)
			break
		}
	}
}

func (s *serverList) RemoveListener(listener peering.Listener) {
	for {
		prev, listeners := s.removeListenerInternal(listener)
		if atomic.CompareAndSwapPointer(&s.listPointer, prev, unsafe.Pointer(&listeners)) {
			break
		}
	}
}

func (s *serverList) removeListenerInternal(listener peering.Listener) (unsafe.Pointer, []peering.Listener) {
	prev := atomic.LoadPointer(&s.listPointer)
	listeners := *(*[]peering.Listener)(prev)
	out := make([]peering.Listener, len(listeners))
	copy(out, listeners)
	for i, k := range out {
		if k == listener {
			out[i] = out[len(out)-1]
			out = out[:len(out)-1]
		}
	}
	return prev, out
}

func (s *serverList) listeners() []peering.Listener {
	return *(*[]peering.Listener)(atomic.LoadPointer(&s.listPointer))
}
