package setup

import (
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/composite"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/empty"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/extjwt"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/fake"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/s2s"
)

func AuthSource(use, method, s2sIssuer, s2sAudience string, s2sSecret s2s.OpaqueBytes, logger logging.Function) stream.AuthSource {
	switch method {
	case "s2s":
		src, err := s2s.NewAuthSource(s2sSecret, s2sIssuer, s2sAudience)
		if err != nil {
			PanicWithMessage("Failed to create s2s "+use+" auth source", err)
		}
		return src
	case "fake":
		return fake.NewAuthSource()
	}
	panic("Unknown auth source method: " + method)
}

func AuthReader(use, methods, s2sAudience string, validatorUrl *url.URL, s2sSecret s2s.OpaqueBytes, metrics metrics.Tracker, logger logging.Function) *auth.Reader {
	decoders := []auth.Decoder{}
	resolvers := []auth.Resolver{}
	for _, method := range strings.Split(methods, ",") {
		dec, res := createAuthHandler(use, strings.TrimSpace(method), s2sAudience, validatorUrl, s2sSecret, metrics, logger)
		decoders = append(decoders, dec)
		resolvers = append(resolvers, res)
	}
	if len(decoders) == 1 {
		return auth.NewReader(decoders[0], resolvers[0])
	}
	return auth.NewReader(composite.NewDecoder(decoders...), composite.NewResolver(resolvers...))
}

func AuthExtractor(methods string) auth.Extractor {
	extractors := []auth.Extractor{}
	for _, method := range strings.Split(methods, ",") {
		extractors = append(extractors, createAuthExtractor(strings.TrimSpace(method)))
	}
	if len(extractors) == 1 {
		return extractors[0]
	}
	return composite.NewExtractor(extractors...)
}

func createAuthExtractor(method string) auth.Extractor {
	switch method {
	case "empty":
		return empty.NewExtractor()
	case "fake-validator", "fake-validator-broken", "validator":
		return extjwt.NewExtractor()
	case "fake":
		return fake.NewExtractor()
	case "s2s":
		return s2s.NewExtractor()
	}
	panic("Unknown auth extractor method: " + method)
}

func createAuthHandler(use, method, s2sAudience string, validatorURL *url.URL, s2sSecret s2s.OpaqueBytes, metrics metrics.Tracker, logger logging.Function) (auth.Decoder, auth.Resolver) {
	switch method {
	case "empty":
		logger(logging.Info, "Installing an empty", use, "auth resolver; accepts anonymous requests")
		return empty.NewDecoder(), empty.NewResolver()
	case "validator":
		logger(logging.Info, "Installing an extjwt (validator)", use, "auth resolver; allows only valid tokens")
		validator, err := extjwt.NewRemoteValidator(validatorURL, http.Client{Timeout: 5 * time.Second}, metrics, logger)
		if err != nil {
			PanicWithMessage("Unable to initialize authorization", err)
		}
		return extjwt.NewDecoder(), extjwt.NewResolver(validator)
	case "fake-validator":
		logger(logging.Info, "Installing a extjwt (fake)", use, "auth resolver; allows valid tokens without verifying the signature")
		return extjwt.NewDecoder(), extjwt.NewResolver(extjwt.NewFakeValidator(100, 0, nil))
	case "fake-validator-broken":
		logger(logging.Info, "Installing a extjwt (fake)", use, "auth resolver; allows valid tokens without verifying the signature")
		return extjwt.NewDecoder(), extjwt.NewResolver(extjwt.NewFakeValidator(50, 50, stream.ErrInvalidSignature))
	case "fake":
		logger(logging.Info, "Installing a fake", use, "auth handler; should allow everything")
		return fake.NewDecoder(), fake.NewResolver(stream.AllPermissions(), nil)
	case "s2s":
		logger(logging.Info, "Installing a shared secret", use, "auth handler; allows only valid tokens")
		res, err := s2s.NewResolver(s2sSecret, s2sAudience)
		if err != nil {
			PanicWithMessage(fmt.Sprintf("Failed to create s2s %s auth resolver", use), err)
		}
		return s2s.NewDecoder(), res
	}
	panic("Unknown auth handler method: " + method)
}
