package stream

import (
	"strconv"
	"strings"
)

// AddressKey describes a unique string describing an address
type AddressKey string

// ToAddress converts a Key back to its Address
func (k AddressKey) ToAddress() (Address, error) {
	return ParseAddress(string(k))
}

// ToAddress converts a Key back to its AddressScope
func (k AddressKey) ToAddressScope() (AddressScope, error) {
	return ParseAddressScope(string(k))
}

// ParseAddress attempts to construct an address from a string
func ParseAddress(key string) (Address, error) {
	versionIndex := strings.IndexByte(key, '@')
	filterIndex := strings.IndexByte(key, '?')

	var namespace string
	switch versionIndex {
	case 0:
		return nil, ErrMissingRequiredNamespace
	case -1:
		return nil, ErrMissingRequiredVersion
	default:
		namespace = key[:versionIndex]
		if strings.ContainsAny(namespace, reserved) { // covers issues like filters before version
			return nil, ErrReservedCharacters
		}
	}
	var version string
	if filterIndex < 0 {
		version = key[versionIndex+1:]
	} else {
		version = key[versionIndex+1 : filterIndex]
	}
	if len(key) == filterIndex+1 { // terminal `?` without filters
		filterIndex = -1
		key = key[:len(key)-1]
	}
	if filterIndex < 0 {
		return StringsToAddress(namespace, version, nil)
	}
	filters := make(map[string]string)
	segments := strings.Split(key[filterIndex+1:], "&")
	if len(segments) > maxFilters {
		return nil, ErrTooManyFilters
	}
	for _, seg := range segments {
		valueIndex := strings.IndexByte(seg, '=')
		if valueIndex < 0 {
			return nil, ErrMissingFilterValue(seg)
		}
		filters[seg[:valueIndex]] = seg[valueIndex+1:]
	}
	return StringsToAddress(namespace, version, filters)
}

// StringsToAddress converts string segments to an address
func StringsToAddress(nspace string, version string, filters map[string]string) (Address, error) {
	v, err := strconv.ParseUint(version, 10, 16)
	if err != nil {
		if err.(*strconv.NumError).Err == strconv.ErrRange {
			return nil, ErrVersionOutOfRange
		}
		return nil, ErrVersionSyntax
	}
	return NewAddress(Namespace(nspace), Version(v), filters)
}
