package stream

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAddressKey(t *testing.T) {
	t.Run("should convert back to an addresss", func(t *testing.T) {
		key := AddressKey("n@1?a=b") // other use cases are tested by ParseAddress
		addr, err := key.ToAddress()
		require.NoError(t, err)
		assert.Equal(t, Namespace("n"), addr.Namespace())
		assert.Equal(t, Version(1), addr.Version())
		val, found := addr.Filter("a")
		assert.True(t, found)
		assert.Equal(t, "b", val)
	})

	t.Run("should convert back to an addresss scope", func(t *testing.T) {
		key := AddressKey("*") // other use cases are tested by ParseAddress
		scope, err := key.ToAddressScope()
		require.NoError(t, err)
		assert.Equal(t, AnyAddress, scope)
	})
}

func TestParseAddress(t *testing.T) {
	nspace := Namespace("n")
	version := Version(1)

	t.Run("should parse a global address", func(t *testing.T) {
		addr, err := ParseAddress("n@1")
		assert.NoError(t, err)
		assert.IsType(t, &globalAddress{}, addr)
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		assert.Equal(t, AddressKey("n@1"), addr.Key())
	})

	t.Run("should parse a 0 filter address", func(t *testing.T) {
		addr, err := ParseAddress("n@1?")
		assert.NoError(t, err)
		assert.IsType(t, &globalAddress{}, addr)
		assert.Equal(t, 1, addr.Cardinality())
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		assert.Equal(t, AddressKey("n@1"), addr.Key())
	})

	t.Run("should parse a 1 filter address", func(t *testing.T) {
		addr, err := ParseAddress("n@1?k=v")
		assert.NoError(t, err)
		assert.IsType(t, &singleFilterAddress{}, addr)
		assert.Equal(t, 2, addr.Cardinality())
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		assert.Equal(t, AddressKey("n@1?k=v"), addr.Key())
		v, ok := addr.Filter("k")
		assert.True(t, ok)
		assert.Equal(t, "v", v)
	})

	t.Run("should parse a 2 filter address", func(t *testing.T) {
		addr, err := ParseAddress("n@1?k=v&a=b")
		assert.NoError(t, err)
		assert.IsType(t, &multiFilterAddress{}, addr)
		assert.Equal(t, 3, addr.Cardinality())
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		// note resort of filters
		assert.Equal(t, AddressKey("n@1?a=b&k=v"), addr.Key())
		v, ok := addr.Filter("k")
		assert.True(t, ok)
		assert.Equal(t, "v", v)
		v, ok = addr.Filter("a")
		assert.True(t, ok)
		assert.Equal(t, "b", v)
	})

	t.Run("should error on illegal namespace", func(t *testing.T) {
		_, err := ParseAddress("n?@1")
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on missing namespace", func(t *testing.T) {
		_, err := ParseAddress("@1")
		assert.Equal(t, ErrMissingRequiredNamespace, err)
	})

	t.Run("should error on missing version", func(t *testing.T) {
		_, err := ParseAddress("n")
		assert.Equal(t, ErrMissingRequiredVersion, err)
	})

	t.Run("should error on illegal version", func(t *testing.T) {
		_, err := ParseAddress("n@q")
		assert.Equal(t, ErrVersionSyntax, err)
	})

	t.Run("should error on out of range version", func(t *testing.T) {
		_, err := ParseAddress("n@100000")
		assert.Equal(t, ErrVersionOutOfRange, err)
	})

	t.Run("should error on illegal field (single optimized)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k@=v")
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on illegal field (multi)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k@=v&a=b")
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on missing value (single optimized)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k")
		assert.Equal(t, ErrMissingFilterValue("k"), err)
	})

	t.Run("should error on missing value (multi)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k&a=b")
		assert.Equal(t, ErrMissingFilterValue("k"), err)
	})

	t.Run("should error on illegal value (single optimized)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k=v@")
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on illegal value (multi)", func(t *testing.T) {
		_, err := ParseAddress("n@1?k=v@&a=b")
		assert.Equal(t, ErrReservedCharacters, err)
	})
}
