package stream

import (
	"sort"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGlobalScope(t *testing.T) {
	addr, err := ParseAddress("n@1")
	require.NoError(t, err)
	assert.Zero(t, AnyAddress.Cardinality())
	assert.True(t, AnyAddress.Includes(addr))
	assert.Equal(t, "^*", AnyAddress.String())
	assert.Equal(t, anyAddressKey, AnyAddress.Key())
	assert.Empty(t, AnyAddress.Parents())
}

func TestParseAddressScope(t *testing.T) {
	t.Run("should return legal addresses when appropriate", func(t *testing.T) {
		expected, err := ParseAddress("n@1")
		require.NoError(t, err)
		actual, err := ParseAddressScope("n@1")
		require.NoError(t, err)
		assert.Equal(t, expected, actual)
	})

	t.Run("should return AnyAddress when appropriate", func(t *testing.T) {
		actual, err := ParseAddressScope(string(anyAddressKey))
		require.NoError(t, err)
		assert.Equal(t, AnyAddress, actual)
	})

	t.Run("should return parse errors when appropriate", func(t *testing.T) {
		actual, err := ParseAddressScope("n")
		require.Nil(t, actual)
		assert.Equal(t, ErrMissingRequiredVersion, err)
	})
}

func TestAddressScopes(t *testing.T) {
	a1, err := ParseAddressScope("a@1")
	require.NoError(t, err)
	assert.True(t, AddressScopes{a1, AnyAddress}.Less(0, 1))
	assert.False(t, AddressScopes{a1, AnyAddress}.Less(1, 0))

	a2, err := ParseAddressScope("a@1?x=y")
	require.NoError(t, err)
	assert.True(t, AddressScopes{a2, AnyAddress}.Less(0, 1))
	assert.False(t, AddressScopes{a2, AnyAddress}.Less(1, 0))

	assert.True(t, AddressScopes{a2, a1}.Less(0, 1))
	assert.False(t, AddressScopes{a2, a1}.Less(1, 0))

	b1, err := ParseAddressScope("b@1")
	require.NoError(t, err)
	assert.True(t, AddressScopes{a1, b1}.Less(0, 1))
	assert.False(t, AddressScopes{a1, b1}.Less(1, 0))

	addrs := AddressScopes{b1, a2, a1, AnyAddress}
	sort.Sort(addrs)
	assert.Equal(t, AddressScopes{a2, a1, b1, AnyAddress}, addrs)

	addrs = AddressScopes{a1, AnyAddress}
	bytes, err := addrs.MarshalBinary()
	assert.Equal(t, []byte{'a', '@', '1', '\000', '*'}, bytes)
	assert.NoError(t, err)

	var out AddressScopes
	assert.NoError(t, out.UnmarshalBinary(bytes))
	assert.Equal(t, addrs, out)

	bytes, err = AddressScopes{}.MarshalBinary()
	assert.Equal(t, []byte{}, bytes)
	assert.NoError(t, err)
	assert.NoError(t, out.UnmarshalBinary(bytes))
	assert.Equal(t, AddressScopes{}, out)

	assert.Equal(t, ErrMissingRequiredVersion, out.UnmarshalBinary([]byte{'?'}))
}

func TestContains(t *testing.T) {
	addr, err := ParseAddress("n@1?a=b&c=d&e=f")
	require.NoError(t, err)

	assert.False(t, AddressScopes{}.Contains(AnyAddress))
	assert.False(t, AddressScopes{addr}.Contains(AnyAddress))
	assert.True(t, AddressScopes{AnyAddress}.Contains(AnyAddress))
	assert.True(t, AddressScopes{addr, AnyAddress}.Contains(AnyAddress))
	assert.True(t, AddressScopes{addr, AnyAddress}.Contains(addr))
}

func TestBestMatch(t *testing.T) {
	addr, err := ParseAddress("n@1?a=b&c=d&e=f")
	require.NoError(t, err)

	t.Run("Should return existing cardinality if no match is found", func(t *testing.T) {
		scopes := AddressScopes{}
		card, found := scopes.HasBetterMatch(addr, -1)
		assert.False(t, found)
		assert.Equal(t, -1, card)

		card, found = scopes.HasBetterMatch(addr, AnyAddress.Cardinality())
		assert.False(t, found)
		assert.Equal(t, AnyAddress.Cardinality(), card)
	})

	t.Run("Should improve best if match is found", func(t *testing.T) {
		scopes := AddressScopes{AnyAddress}
		card, found := scopes.HasBetterMatch(addr, -1)
		assert.True(t, found)
		assert.Equal(t, AnyAddress.Cardinality(), card)
	})

	t.Run("Should not declare a match in the case of a tie", func(t *testing.T) {
		scopes := AddressScopes{AnyAddress}
		card, found := scopes.HasBetterMatch(addr, AnyAddress.Cardinality())
		assert.False(t, found)
		assert.Equal(t, AnyAddress.Cardinality(), card)
	})

	t.Run("Should find the best match available if sorted", func(t *testing.T) {
		scopes := addr.Parents()
		scopes = append(scopes, AnyAddress)
		card, found := scopes.HasBetterMatch(addr, -1)
		assert.True(t, found)
		assert.Equal(t, scopes[0].Cardinality(), card)
	})
}

func TestAncestorList(t *testing.T) {
	t.Run("Should return nothing for the base scope", func(t *testing.T) {
		assert.Equal(t, AddressScopes{}, AncestorList(AnyAddress))
	})

	t.Run("Should return a sorted list for other addresses", func(t *testing.T) {
		addr, err := ParseAddress("ext@1?k=v&a=b")
		require.NoError(t, err)
		p1, err := ParseAddress("ext@1?a=b")
		require.NoError(t, err)
		p2, err := ParseAddress("ext@1?k=v")
		require.NoError(t, err)
		p3, err := ParseAddress("ext@1")
		require.NoError(t, err)
		assert.Equal(t, AddressScopes{p1, p2, p3, AnyAddress}, AncestorList(addr))
	})
}
