package stream

import (
	"bytes"
	"encoding/binary"
	"sort"
)

type AddressSourceMap map[AddressKey]SourceID

func (a AddressSourceMap) MarshalBinary() ([]byte, error) {
	data := make([][]byte, len(a))
	i := 0
	for key, value := range a {
		bytes := make([]byte, len(key)+4)
		binary.BigEndian.PutUint32(bytes, uint32(value))
		copy(bytes[4:], key)
		data[i] = bytes
		i++
	}
	return bytes.Join(data, scopeDelimiter), nil
}

func (a *AddressSourceMap) UnmarshalBinary(data []byte) error {
	out := make(AddressSourceMap)
	var key AddressKey
	for len(data) > 0 {
		if len(data) < 5 {
			return ErrMissingRequiredNamespace
		}
		id := SourceID(binary.BigEndian.Uint32(data))
		if end := bytes.Index(data[4:], scopeDelimiter); end != -1 {
			end += 4
			key = AddressKey(data[4:end])
			data = data[end+1:]
		} else {
			key = AddressKey(data[4:])
			data = nil
		}
		if _, err := key.ToAddressScope(); err != nil {
			return err
		}
		out[key] = id
	}
	*a = out
	return nil
}

func (a AddressSourceMap) ToScopes() (AddressScopes, error) {
	if a == nil {
		return nil, nil
	}
	scopes := make(AddressScopes, len(a))
	i := 0
	var err error
	for k := range a {
		if scopes[i], err = k.ToAddressScope(); err != nil {
			return nil, err
		}
		i++
	}
	sort.Sort(scopes)
	return scopes, nil
}
