package stream

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAddressSourceMap_MarshalBinary(t *testing.T) {
	t.Run("should serialize empty set correctly", func(t *testing.T) {
		a := make(AddressSourceMap)
		bytes, err := a.MarshalBinary()
		require.NoError(t, err)
		var out AddressSourceMap
		require.NoError(t, out.UnmarshalBinary(bytes))
		assert.Equal(t, a, out)
	})

	t.Run("should serialize values correctly", func(t *testing.T) {
		a := make(AddressSourceMap)
		a[AnyAddress.Key()] = None
		a[AddressKey("n@1")] = SourceID(1)
		a[AddressKey("x@2?a=b")] = SourceID(0xFFFFFFFF)
		bytes, err := a.MarshalBinary()
		require.NoError(t, err)
		var out AddressSourceMap
		require.NoError(t, out.UnmarshalBinary(bytes))
		assert.Equal(t, a, out)
	})

	t.Run("unmarshal should error on truncated source id", func(t *testing.T) {
		bytes := []byte("\000\000\000")
		var out AddressSourceMap
		assert.Equal(t, ErrMissingRequiredNamespace, out.UnmarshalBinary(bytes))
	})

	t.Run("unmarshal should error on invalid key", func(t *testing.T) {
		bytes := []byte("\000\000\000\000n?a=b")
		var out AddressSourceMap
		assert.Equal(t, ErrMissingRequiredVersion, out.UnmarshalBinary(bytes))
	})
}

func TestAddressSourceMap_ToSources(t *testing.T) {
	addr, err := StringsToAddress("n", "1", nil)
	require.NoError(t, err)
	addr2, err := StringsToAddress("x", "2", map[string]string{"a": "b"})
	require.NoError(t, err)

	t.Run("should convert empty set correctly", func(t *testing.T) {
		scopes, err := AddressSourceMap(nil).ToScopes()
		assert.Nil(t, scopes)
		assert.NoError(t, err)
		scopes, err = AddressSourceMap{}.ToScopes()
		assert.Empty(t, scopes)
		assert.NoError(t, err)
	})

	t.Run("should convert values correctly", func(t *testing.T) {
		a := make(AddressSourceMap)
		a[addr.Key()] = SourceID(1)
		a[addr2.Key()] = SourceID(0xFFFFFFFF)
		scopes, err := a.ToScopes()
		require.NoError(t, err)
		assert.Equal(t, AddressScopes{addr2, addr}, scopes)
	})

	t.Run("should error on conversion of bad values", func(t *testing.T) {
		a := make(AddressSourceMap)
		a[addr.Key()] = SourceID(1)
		a[AddressKey("n")] = SourceID(0xFFFFFFFF)
		scopes, err := a.ToScopes()
		assert.Nil(t, scopes)
		assert.Equal(t, ErrMissingRequiredVersion, err)
	})
}
