package stream

import (
	"fmt"
	"strconv"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestNewAddress(t *testing.T) {
	nspace := Namespace("n")
	version := Version(1)
	field1 := "k"
	value1 := "v"
	field2 := "a"
	value2 := "b"

	t.Run("should construct a global address", func(t *testing.T) {
		addr, err := NewAddress(nspace, version, nil)
		assert.NoError(t, err)
		assert.IsType(t, &globalAddress{}, addr)
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		assert.Equal(t, AddressKey(fmt.Sprintf("%v@%v", nspace, version)), addr.Key())
	})

	t.Run("should parse a 1 filter address", func(t *testing.T) {
		addr, err := NewAddress(nspace, version, map[string]string{field1: value1})
		assert.NoError(t, err)
		assert.IsType(t, &singleFilterAddress{}, addr)
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		assert.Equal(t, AddressKey(fmt.Sprintf("%v@%v?%v=%v", nspace, version, field1, value1)), addr.Key())
		v, ok := addr.Filter(field1)
		assert.True(t, ok)
		assert.Equal(t, value1, v)
	})

	t.Run("should parse a 2 filter address", func(t *testing.T) {
		addr, err := NewAddress(nspace, version, map[string]string{field1: value1, field2: value2})
		assert.NoError(t, err)
		assert.IsType(t, &multiFilterAddress{}, addr)
		assert.Equal(t, nspace, addr.Namespace())
		assert.Equal(t, version, addr.Version())
		// note resorting so a < k in fields
		assert.Equal(t, AddressKey(fmt.Sprintf("%v@%v?%v=%v&%v=%v", nspace, version, field2, value2, field1, value1)), addr.Key())
		v, ok := addr.Filter(field1)
		assert.True(t, ok)
		assert.Equal(t, value1, v)
		v, ok = addr.Filter(field2)
		assert.True(t, ok)
		assert.Equal(t, value2, v)
	})

	t.Run("should error on illegal namespace", func(t *testing.T) {
		_, err := NewAddress("n?", version, nil)
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on missing namespace", func(t *testing.T) {
		_, err := NewAddress("", version, nil)
		assert.Equal(t, ErrMissingRequiredNamespace, err)
	})

	t.Run("should error on illegal field (single optimized)", func(t *testing.T) {
		_, err := NewAddress(nspace, version, map[string]string{"k?": value1})
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on illegal field (multi)", func(t *testing.T) {
		_, err := NewAddress(nspace, version, map[string]string{"k?": value1, field2: value2})
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on illegal value (single optimized)", func(t *testing.T) {
		_, err := NewAddress(nspace, version, map[string]string{field1: "v?"})
		assert.Equal(t, ErrReservedCharacters, err)
	})

	t.Run("should error on illegal value (multi)", func(t *testing.T) {
		_, err := NewAddress(nspace, version, map[string]string{field1: "v?", field2: value2})
		assert.Equal(t, ErrReservedCharacters, err)
	})
}

func TestGlobalAddress(t *testing.T) {
	nspace := Namespace("n")
	version := Version(1)
	field1 := "k"
	value1 := "v"

	addr := &globalAddress{nspace, version, "n@1"}
	assert.Equal(t, nspace, addr.Namespace())
	assert.Equal(t, version, addr.Version())
	_, ok := addr.Filter(field1)
	assert.False(t, ok)
	assert.Equal(t, AddressKey("n@1"), addr.Key())
	assert.Equal(t, "^n@1", addr.String())
	assert.True(t, addr.Includes(addr))

	t.Run("parents should match anyaddress", func(t *testing.T) {
		assert.Equal(t, anyAddressSlice, addr.Parents())
	})

	t.Run("should allow adding fields", func(t *testing.T) {
		child, err := addr.WithFilter("key", "value")
		require.NoError(t, err)
		assert.Equal(t, addr.Cardinality()+1, child.Cardinality())
		value, ok := child.Filter("key")
		assert.True(t, ok)
		assert.Equal(t, "value", value)
	})

	t.Run("include should match self", func(t *testing.T) {
		assert.True(t, addr.Includes(addr))
	})

	t.Run("include should match filtered addresses", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value1})
		require.NoError(t, err)
		assert.True(t, addr.Includes(other))
	})

	t.Run("include should not match wrong version", func(t *testing.T) {
		other, err := NewAddress(nspace, Version(2), nil)
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match wrong address", func(t *testing.T) {
		other, err := NewAddress(Namespace("n2"), version, nil)
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})
}

func TestSingleFilterAddress(t *testing.T) {
	nspace := Namespace("n")
	version := Version(1)
	field1 := "k"
	value1 := "v"
	field2 := "a"
	value2 := "b"

	addr := &singleFilterAddress{nspace, version, field1, value1, "n@1?k=v"}
	assert.Equal(t, nspace, addr.Namespace())
	assert.Equal(t, version, addr.Version())
	v, ok := addr.Filter(field1)
	assert.True(t, ok)
	assert.Equal(t, value1, v)
	_, ok = addr.Filter(value1)
	assert.False(t, ok)
	assert.Equal(t, AddressKey("n@1?k=v"), addr.Key())
	assert.Equal(t, "^n@1?k=v", addr.String())
	assert.True(t, addr.Includes(addr))

	t.Run("should allow adding fields", func(t *testing.T) {
		child, err := addr.WithFilter("key", "value")
		require.NoError(t, err)
		assert.Equal(t, addr.Cardinality()+1, child.Cardinality())
		value, ok := child.Filter("key")
		assert.True(t, ok)
		assert.Equal(t, "value", value)
	})

	t.Run("include should match self", func(t *testing.T) {
		assert.True(t, addr.Includes(addr))
	})

	t.Run("parents should match global address", func(t *testing.T) {
		other, err := NewAddress(nspace, version, nil)
		require.NoError(t, err)
		assert.Equal(t, AddressScopes{other}, addr.Parents())
	})

	t.Run("include should not match global addresses", func(t *testing.T) {
		other, err := NewAddress(nspace, version, nil)
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should match filtered addresses", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value1, field2: value2})
		require.NoError(t, err)
		assert.True(t, addr.Includes(other))
	})

	t.Run("include should not match different filter", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value2})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match wrong version", func(t *testing.T) {
		other, err := NewAddress(nspace, Version(2), map[string]string{field1: value1})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match wrong address", func(t *testing.T) {
		other, err := NewAddress(Namespace("n2"), version, map[string]string{field1: value1})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include allow a limited number of filters", func(t *testing.T) {
		filters := make(map[string]string)
		segments := []string{}
		for i := int64(0); i <= maxFilters+10; i++ {
			key := strconv.FormatInt(i, 10)
			value := "value"
			filters[key] = value
			segments = append(segments, key+"="+value)
		}
		_, err := NewAddress(nspace, version, filters)
		assert.Equal(t, ErrTooManyFilters, err)

		_, err = ParseAddress("n@1?" + strings.Join(segments, "&"))
		assert.Equal(t, ErrTooManyFilters, err)
	})
}

func TestMultiFilterAddress(t *testing.T) {
	nspace := Namespace("n")
	version := Version(1)
	field1 := "k"
	value1 := "v"
	field2 := "a"
	value2 := "b"

	addr := newMultiFilterAddress(nspace, version, map[string]string{field1: value1, field2: value2})
	assert.Equal(t, nspace, addr.Namespace())
	assert.Equal(t, version, addr.Version())
	v, ok := addr.Filter(field1)
	assert.True(t, ok)
	assert.Equal(t, value1, v)
	v, ok = addr.Filter(field2)
	assert.True(t, ok)
	assert.Equal(t, value2, v)
	_, ok = addr.Filter(value1)
	assert.False(t, ok)
	assert.Equal(t, AddressKey("n@1?a=b&k=v"), addr.Key())
	assert.Equal(t, "^n@1?a=b&k=v", addr.String())
	assert.True(t, addr.Includes(addr))

	t.Run("should allow adding fields", func(t *testing.T) {
		child, err := addr.WithFilter("key", "value")
		require.NoError(t, err)
		assert.Equal(t, addr.Cardinality()+1, child.Cardinality())
		value, ok := child.Filter("key")
		assert.True(t, ok)
		assert.Equal(t, "value", value)
	})

	t.Run("should have N-1 parents should of Cardinality N-1 at C=3", func(t *testing.T) {
		parents := addr.Parents()
		assert.Len(t, parents, addr.Cardinality()-1)
		for _, parent := range parents {
			assert.True(t, parent.Includes(addr))
			assert.Equal(t, addr.Cardinality()-1, parent.Cardinality())
		}
	})

	t.Run("should have N-1 parents should of Cardinality N-1 at C=4", func(t *testing.T) {
		addr4 := newMultiFilterAddress(nspace, version, map[string]string{field1: value1, field2: value2, "c": "d"})
		parents := addr4.Parents()
		assert.Len(t, parents, addr4.Cardinality()-1)
		for _, parent := range parents {
			assert.True(t, parent.Includes(addr4))
			assert.Equal(t, addr4.Cardinality()-1, parent.Cardinality())
		}
	})

	t.Run("include should match self", func(t *testing.T) {
		assert.True(t, addr.Includes(addr))
	})

	t.Run("include should match extended filters", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value1, field2: value2, value1: value2})
		require.NoError(t, err)
		assert.True(t, addr.Includes(other))
	})

	t.Run("include should not match global addresses", func(t *testing.T) {
		other, err := NewAddress(nspace, version, nil)
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include not should match partial filters", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value1})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match different filter", func(t *testing.T) {
		other, err := NewAddress(nspace, version, map[string]string{field1: value2, field2: value2})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match wrong version", func(t *testing.T) {
		other, err := NewAddress(nspace, Version(2), map[string]string{field1: value1, field2: value2})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})

	t.Run("include should not match wrong address", func(t *testing.T) {
		other, err := NewAddress(Namespace("n2"), version, map[string]string{field1: value1, field2: value2})
		require.NoError(t, err)
		assert.False(t, addr.Includes(other))
	})
}
