package audience

import (
	"fmt"
	"sync"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol/message"
)

// represents current listener state for client
type clientListener struct {
	client    session.Client
	version   protocol.Version
	positions map[stream.AddressKey]*stream.Tracker
	subs      map[stream.AddressKey]stream.Address
	mutex     sync.RWMutex
	logger    logging.Function
	closed    bool
}

var _ stream.Listener = &clientListener{}

func newListener(client session.Client, version protocol.Version, logger logging.Function) *clientListener {
	return &clientListener{
		client:    client,
		version:   version,
		positions: make(map[stream.AddressKey]*stream.Tracker),
		subs:      make(map[stream.AddressKey]stream.Address),
		logger:    logger,
	}
}

func (l *clientListener) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	l.mutex.RLock()
	pos, ok := l.positions[addr.Key()]
	closed := l.closed
	l.mutex.RUnlock()
	if !ok && closed {
		return stream.None, stream.Closed
	}
	return pos.Current()
}

func (l *clientListener) OnDataLost(desc stream.MessageDescription) bool {
	err := error(nil)
	lost, ok := desc.(message.Lost)
	if !ok {
		lost, err = message.NewLost(desc.Address(), desc.Source(), desc.At())
	}
	return l.pos(desc.Address()).Accept(desc) && l.respond(lost, err)
}

func (l *clientListener) OnDataReceived(msg stream.Message) bool {
	err := error(nil)
	sent, ok := msg.(message.Sent)
	if !ok {
		sent, err = message.NewSent(msg.Address(), msg.Source(), msg.At(), msg.Data())
	}
	return l.pos(msg.Address()).Accept(msg) && l.respond(sent, err)
}

func (l *clientListener) OnStreamClosed(addr stream.Address, err error) bool {
	key := addr.Key()
	l.mutex.Lock()
	pos, ok := l.positions[key]
	if ok {
		delete(l.positions, key)
	}
	l.mutex.Unlock()
	src, at := pos.Current()
	return l.respond(message.NewClosed(addr, src, at))
}

func (l *clientListener) subscribe(msg message.Join) error {
	addr := msg.Address()
	key := addr.Key()
	l.mutex.Lock()
	defer l.mutex.Unlock()
	if l.closed {
		return stream.ErrReaderAlreadyClosed
	}
	l.subs[key] = addr
	pos, ok := l.positions[key]
	if !ok {
		pos = new(stream.Tracker)
	}
	pos.Set(msg.Source(), msg.Position())
	return nil
}

func (l *clientListener) unsubscribe(addr stream.Address) {
	key := addr.Key()
	l.mutex.Lock()
	defer l.mutex.Unlock()
	delete(l.subs, key)
}

func (l *clientListener) listInvalidated(creds stream.Credentials) []stream.Address {
	toRemove := []stream.Address{}
	l.mutex.RLock()
	defer l.mutex.RUnlock()
	for _, addr := range l.subs {
		if !creds.CanListen(addr) {
			toRemove = append(toRemove, addr)
		}
	}
	return toRemove
}

func (l *clientListener) pos(addr stream.Address) *stream.Tracker {
	key := addr.Key()
	l.mutex.Lock()
	defer l.mutex.Unlock()
	pos, ok := l.positions[key]
	if !ok && !l.closed {
		pos = new(stream.Tracker)
		l.positions[key] = pos
	}
	return pos
}

func (l *clientListener) respond(msg protocol.Message, srcErr error) bool {
	var err error
	if srcErr != nil {
		msg, err = message.NewError(protocol.NoRequest, srcErr)
	}
	if err == nil {
		err = l.respondInternal(msg)
	}
	if err != nil {
		l.client.Close(err) // TODO : wrap these error closes in a common handler
		return false
	}
	return true
}

func (l *clientListener) respondInternal(msg protocol.Message) error {
	if !l.version.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if l.version < protocol.Five && msg.OpCode() == protocol.Move {
		// backward compatability -- cause complete reconnection
		// instead of moving one address
		msg, _ = message.NewDrain()
	}
	bytes, err := msg.Marshal(l.version)
	if err != nil {
		return err
	}
	l.logger(logging.Trace, "Aud [SND]=>", l.client.Address(), msg)
	return l.client.WriteBinary(bytes)
}

func (l *clientListener) onClosed() []stream.AddressKey {
	out := []stream.AddressKey{}
	l.mutex.Lock()
	defer l.mutex.Unlock()
	l.closed = true
	for key := range l.subs {
		out = append(out, key)
	}
	l.subs = make(map[stream.AddressKey]stream.Address)
	return out
}

func (l *clientListener) String() string {
	return fmt.Sprintf(`{listener: %s}`, l.client.Address())
}
