package audience

import (
	"sync"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

type audienceWriter interface {
	stream.Writer
	send([]byte, bool, stream.MutableTrackerPromise)
	respond(protocol.Message, error)
}

type writer struct {
	ch      *channel
	binding *binding
	closed  bool
	mutex   sync.RWMutex
}

var _ audienceWriter = (*writer)(nil)

func newWriter(ch *channel, b *binding) *writer { return &writer{ch: ch, binding: b} }

func (w *writer) Address() stream.Address { return w.ch.Address() }
func (w *writer) Close() error {
	w.mutex.Lock()
	closed := w.closed
	w.closed = true
	w.mutex.Unlock()
	if closed {
		return stream.ErrWriterAlreadyClosed
	}
	return w.ch.closeWriter(w)
}

func (w *writer) respond(msg protocol.Message, err error) {
	w.binding.respond(msg, err)
}

func (w *writer) Send(message []byte, isDelta bool) stream.TrackerPromise {
	p := stream.NewTrackerPromise()
	w.send(message, isDelta, stream.NewTrackerPromise())
	return p
}

func (w *writer) send(message []byte, isDelta bool, promise stream.MutableTrackerPromise) {
	w.mutex.RLock()
	defer w.mutex.RUnlock()
	w.ch.send(message, isDelta, promise)
}

type movedWriter struct{ addr stream.Address }

var _ audienceWriter = (*movedWriter)(nil)

func (m *movedWriter) Address() stream.Address { return m.addr }
func (m *movedWriter) Close() error            { return nil }
func (w *movedWriter) Send(message []byte, isDelta bool) stream.TrackerPromise {
	p := stream.NewTrackerPromise()
	w.send(message, isDelta, stream.NewTrackerPromise())
	return p
}

func (m *movedWriter) respond(msg protocol.Message, err error) {}
func (m *movedWriter) send(message []byte, isDelta bool, promise stream.MutableTrackerPromise) {
	promise.Set(stream.Untracked, protocol.ErrAddressMoved)
}
