package composite

import (
	"net/http"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/empty"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/extjwt"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/fake"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestExtractor(t *testing.T) {
	t.Run("should return first match if possible (extract)", func(t *testing.T) {
		ext := NewExtractor(empty.NewExtractor(), fake.NewExtractor())
		assert.Equal(t, "empty,fake", ext.Method())
		req, ok := ext.Extract(http.Header{})
		assert.True(t, ok)
		require.NotNil(t, req)
		assert.Equal(t, "empty", req.Method())

		ext = NewExtractor(fake.NewExtractor(), empty.NewExtractor())
		assert.Equal(t, "fake,empty", ext.Method())
		req, ok = ext.Extract(http.Header{})
		assert.True(t, ok)
		require.NotNil(t, req)
		assert.Equal(t, "fake", req.Method())
	})

	t.Run("should iterate past failed matches", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, "token") // not empty

		ext := NewExtractor(empty.NewExtractor(), fake.NewExtractor())
		req, ok := ext.Extract(header)
		assert.True(t, ok)
		require.NotNil(t, req)
		assert.Equal(t, "fake", req.Method())
	})

	t.Run("should error if nothing matches (empty case)", func(t *testing.T) {
		ext := NewExtractor()
		assert.Equal(t, "", ext.Method())

		req, ok := ext.Extract(http.Header{})
		assert.False(t, ok)
		assert.Nil(t, req)
		assert.Equal(t, stream.ErrUnsupportedAuthMethod, ext.FailureReason(http.Header{}))
	})

	t.Run("should forward first failure reason if nothing matches", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, "token") // not empty or jwt

		jwt := extjwt.NewExtractor()
		ext := NewExtractor(jwt, empty.NewExtractor())

		req, ok := ext.Extract(header)
		assert.False(t, ok)
		assert.Nil(t, req)
		assert.Equal(t, jwt.FailureReason(header), ext.FailureReason(header))
	})
}
