package extjwt

import (
	"sort"

	"code.justin.tv/common/jwt"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	channelKey     = "c"
	extKey         = "e"
	audKey         = "a"
	topicBroadcast = "broadcast"
	topicGlobal    = "global"
	topicAny       = "*"
	extNamespace   = stream.Namespace("ext")
	extVersion     = stream.Version(1)
)

// Note : omit fields we don't need here
type claims struct {
	ChannelID string              `json:"channel_id"`
	Verbs     map[string][]string `json:"pubsub_perms"`
	Expires   int64               `json:"exp"`
}

func (c *claims) toScopes(clientID, verb string) stream.AddressScopes {
	perms := c.Verbs[verb]
	scopes := make(stream.AddressScopes, 0, len(perms))
	for _, v := range perms {
		addr, err := buildAddress(c.ChannelID, clientID, v)
		if err != nil {
			continue
		}
		// optimize out redundant scopes
		found := false
		for i, a := range scopes {
			if a.Includes(addr) {
				found = true
			}
			if cast, ok := a.(stream.Address); ok && addr.Includes(cast) {
				scopes[i] = addr
				found = true
			}
		}
		if !found {
			scopes = append(scopes, addr)
		}
	}
	sort.Sort(scopes)
	return scopes
}

func buildAddress(channelID, clientID, v string) (stream.Address, error) {
	filters := make(map[string]string)
	filters[extKey] = clientID
	filters[audKey] = "*"
	switch v {
	case topicBroadcast:
		filters[channelKey] = channelID
	case topicGlobal:
		filters[channelKey] = "*"
	case topicAny:
	default:
		return nil, protocol.ErrInvalidAddress
	}
	return stream.NewAddress(extNamespace, extVersion, filters)
}

func parse(token OpaqueBytes) (*claims, error) {
	f, err := jwt.Parse(token)
	if err != nil {
		return nil, stream.ErrInvalidJWT
	}

	var out claims
	var header struct{} // we throw this out
	if err = f.Decode(&header, &out); err != nil {
		return nil, err
	}

	return &out, nil
}
