package extjwt

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
)

const (
	channelID = "12345"
	clientID  = "abcd"
)

var (
	anyChannelAddr, _ = stream.NewAddress(extNamespace, extVersion, map[string]string{audKey: "*", extKey: clientID})
	channelAddr, _    = stream.NewAddress(extNamespace, extVersion, map[string]string{audKey: "*", extKey: clientID, channelKey: channelID})
	globalAddr, _     = stream.NewAddress(extNamespace, extVersion, map[string]string{audKey: "*", extKey: clientID, channelKey: "*"})
	channel2Addr, _   = stream.NewAddress(extNamespace, extVersion, map[string]string{audKey: "*", extKey: clientID, channelKey: "67890"})
	ext2Addr, _       = stream.NewAddress(extNamespace, extVersion, map[string]string{audKey: "*", extKey: "xyz", channelKey: channelID})
)

func TestClaims(t *testing.T) {
	t.Run("should convert '*' to any-channel address", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"*"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{anyChannelAddr}, scopes)
	})

	t.Run("should convert 'broadcast' to channel address", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"broadcast"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{channelAddr}, scopes)
	})

	t.Run("should convert 'global' to global address", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"global"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{globalAddr}, scopes)
	})

	t.Run("should convert 'global' + 'broadcast' to both", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"broadcast", "global"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{globalAddr, channelAddr}, scopes)
	})

	t.Run("should ignore unknown types", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"unknown"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{}, scopes)
	})

	t.Run("should merge overlapping types", func(t *testing.T) {
		scopes := (&claims{
			ChannelID: channelID,
			Verbs:     map[string][]string{listenVerb: []string{"global", "*", "broadcast"}},
		}).toScopes(clientID, listenVerb)
		assert.Equal(t, stream.AddressScopes{anyChannelAddr}, scopes)
	})
}

func TestParse(t *testing.T) {
	t.Run("should return original claims", func(t *testing.T) {
		claims := &claims{channelID, map[string][]string{"verb": []string{"value"}}, 1}
		out, err := parse(testToken(claims))
		assert.NoError(t, err)
		assert.Equal(t, claims, out)
	})

	t.Run("should report broken tokens", func(t *testing.T) {
		out, err := parse(OpaqueBytes("broker"))
		assert.Nil(t, out)
		assert.Equal(t, stream.ErrInvalidJWT, err)
	})
}
