package extjwt

import (
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/common/jwt"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

func testToken(c *claims) OpaqueBytes {
	token, _ := jwt.Encode(struct{}{}, c, jwt.None)
	return OpaqueBytes(token)
}

func TestExtractor(t *testing.T) {
	ext := NewExtractor()
	validToken := testToken(&claims{})
	validAuth := protocol.HTTPBearerPrefix + string(validToken)

	t.Run("should allow valid tokens", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, validAuth)
		header.Set(protocol.HTTPClientIDHeader, clientID)
		req, ok := ext.Extract(header)
		require.NotNil(t, req)
		assert.Equal(t, &request{clientID, validToken}, req)
		assert.True(t, ok)
		assert.NoError(t, ext.FailureReason(header))
	})

	t.Run("should report its method correctly", func(t *testing.T) {
		assert.Equal(t, method, ext.Method())
	})

	t.Run("should error when no Authorization is provided", func(t *testing.T) {
		header := http.Header{}
		_, ok := ext.Extract(header)
		assert.False(t, ok)
		assert.Equal(t, stream.ErrMissingAuthorization, ext.FailureReason(header))
	})

	t.Run("should error when malformed Authorization is provided", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, "token")
		header.Set(protocol.HTTPClientIDHeader, clientID)
		req, ok := ext.Extract(header)
		assert.Nil(t, req)
		assert.False(t, ok)
		assert.Equal(t, stream.ErrInvalidAuthHeaders, ext.FailureReason(header))
	})

	t.Run("should error when no Client-ID is provided", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, validAuth)
		req, ok := ext.Extract(header)
		assert.Nil(t, req)
		assert.False(t, ok)
		assert.Equal(t, stream.ErrMissingClientID, ext.FailureReason(header))
	})

	t.Run("should error when malformed token is provided", func(t *testing.T) {
		header := http.Header{}
		header.Set(protocol.HTTPAuthorizationHeader, protocol.HTTPBearerPrefix+"token")
		header.Set(protocol.HTTPClientIDHeader, clientID)
		req, ok := ext.Extract(header)
		assert.Nil(t, req)
		assert.False(t, ok)
		assert.Equal(t, stream.ErrInvalidJWT, ext.FailureReason(header))
	})
}
