package extjwt

import (
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	httpx "code.justin.tv/devhub/e2ml/libs/http"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

type remoteValidator struct {
	client http.Client
	url    *url.URL
	stats  metrics.Tracker
	logger logging.Function
}

// NewRemoteValidator attempts to connect to a remote extjwt validator service
// to validate tokens
func NewRemoteValidator(url *url.URL, client http.Client, stats metrics.Tracker, logger logging.Function) (Validator, error) {
	if err := healthCheck(url, client); err != nil {
		return nil, fmt.Errorf("Unable to reach Validator: %v", err)
	}
	// perform health check
	return remoteValidator{client, url, stats, logger}.validate, nil
}

func (v remoteValidator) validate(clientID string, jwt OpaqueBytes, claims *claims) (bool, error) {
	if claims == nil {
		return false, stream.ErrMissingJWTClaims
	}

	if err := validateClient(clientID); err != nil {
		return false, err
	}

	if err := validateChannel(claims.ChannelID); err != nil {
		return false, err
	}

	if time.Unix(claims.Expires, 0).Before(time.Now()) {
		return false, stream.ErrAuthExpired
	}

	listen, ok := claims.Verbs[listenVerb]
	if !ok || len(listen) < 1 {
		return false, stream.ErrMissingListenPermissions
	}

	reqURL := *v.url
	reqURL.Path = "/client/" + clientID + "/permission/listen"
	query := make(url.Values)
	query.Set("channel", claims.ChannelID)
	query.Set("targets", listen[0])
	reqURL.RawQuery = query.Encode()

	req, _ := http.NewRequest("GET", reqURL.String(), nil)
	req.Header.Set(protocol.HTTPAuthorizationHeader, protocol.HTTPBearerPrefix+string(jwt))

	resp, err := v.client.Do(req)
	if resp != nil && resp.Body != nil {
		defer resp.Body.Close()
	}
	if err != nil {
		v.logger(logging.Debug, "Validator unreachable:", err)
		return false, protocol.ErrServiceUnavailable
	}

	// TODO : metrics on response codes
	if httpErr, found := httpx.ExtractErrorFromResponse(resp); found {
		return false, httpErr
	}
	return true, nil
}

func healthCheck(baseURL *url.URL, client http.Client) error {
	reqURL := *baseURL
	reqURL.Path = "/debug/running"
	req, _ := http.NewRequest("GET", reqURL.String(), nil)
	resp, err := client.Do(req)
	if resp != nil && resp.Body != nil {
		defer resp.Body.Close()
	}
	if err != nil {
		return err
	}
	if httpErr, found := httpx.ExtractErrorFromResponse(resp); found {
		return httpErr
	}
	return nil
}

// https://git-aws.internal.justin.tv/web/owl/blob/master/oauth2/oauth2.go#L31
// https://git-aws.internal.justin.tv/web/owl/blob/master/internal/util/tokengen.go#L8
func validateClient(clientID string) error {
	if clientID == "" {
		return stream.ErrMissingClientID
	}

	if len(clientID) > 40 { // give some slop for possible expansion
		return stream.ErrInvalidClientID
	}

	for _, r := range clientID {
		if (r < '0' || r > '9') && (r < 'a' || r > 'z') {
			return stream.ErrInvalidClientID
		}
	}
	return nil
}

func validateChannel(channelID string) error {
	if channelID == "" {
		return stream.ErrMissingChannelID
	}

	if _, err := strconv.ParseUint(channelID, 10, 64); err != nil {
		return stream.ErrInvalidChannelID
	}
	return nil
}
