package s2s

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestNewAuthSource(t *testing.T) {
	s, err := NewAuthSource(newValidSecret(), "iss-me", "aud-you")
	require.NoError(t, err)
	require.NotNil(t, s)
}

func TestNewAuthSource_InvalidSecret(t *testing.T) {
	_, err := NewAuthSource(newSecretWithLen(1), "iss-me", "aud-you")
	require.EqualError(t, err, "s2s-secret is empty or has less than 64 bytes")
}

func TestAuthSource_Next(t *testing.T) {
	req := newFakeRequest(t, newValidSecret(), "aud-you")
	assert.NotEmpty(t, req.encodedJWT, "req.encodedJWT")
}

func TestValidateSecretLen(t *testing.T) {
	err := ValidateSecretLen([]byte(newSecretWithLen(0)))
	require.EqualError(t, err, "s2s-secret is empty or has less than 64 bytes")
	err = ValidateSecretLen([]byte(newSecretWithLen(63)))
	require.EqualError(t, err, "s2s-secret is empty or has less than 64 bytes")

	err = ValidateSecretLen([]byte(newSecretWithLen(64)))
	require.NoError(t, err)
	err = ValidateSecretLen([]byte(newSecretWithLen(65)))
	require.NoError(t, err)
}

// test helpers

func newFakeRequest(t *testing.T, secret stream.OpaqueBytes, aud string) *request {
	s, err := NewAuthSource(secret, "iss-me", aud)
	require.NoError(t, err)
	require.NotNil(t, s)

	req, ok := s().(*request)
	require.True(t, ok, "request type s2s.request")
	return req
}

func newValidSecret() OpaqueBytes {
	return newSecretWithLen(64)
}

func newSecretWithLen(length int) OpaqueBytes {
	s := ""
	for i := 0; i < length; i++ {
		s += "s"
	}
	return OpaqueBytes(s)
}
