package s2s

import (
	"net/http"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"

	"github.com/stretchr/testify/require"
)

func TestExtractor_Method(t *testing.T) {
	ext := NewExtractor()
	require.Equal(t, s2sMethod, ext.Method())
}

func TestExtractor_Extract_ValidToken(t *testing.T) {
	ext := NewExtractor()
	r := newFakeRequest(t, newValidSecret(), "aud-you")
	tokenStr := string(r.encodedJWT)
	h := http.Header{"Authorization": {"Bearer " + tokenStr}}

	authReq, ok := ext.Extract(h)
	require.True(t, ok, "ext.Extract ok")

	req, ok := authReq.(*request)
	require.True(t, ok, "request type s2s.Request")
	require.Equal(t, string(req.encodedJWT), tokenStr)
	require.NotNil(t, req.parsedJWT, "parsedJWT")
	require.NoError(t, ext.FailureReason(h))
}

func TestExtractor_Extract_MissingHeader(t *testing.T) {
	ext := NewExtractor()
	h := http.Header{} // no auth header

	authReq, ok := ext.Extract(h)
	require.False(t, ok, "ext.Extract ok")
	require.Nil(t, authReq)
	require.EqualError(t, ext.FailureReason(h), stream.ErrMissingAuthorization.Error())
}

func TestExtractor_Extract_MissingBearerPrefix(t *testing.T) {
	ext := NewExtractor()
	r := newFakeRequest(t, newValidSecret(), "aud-you")
	tokenStr := string(r.encodedJWT)
	h := http.Header{"Authorization": {tokenStr}} // no Bearer prefix

	authReq, ok := ext.Extract(h)
	require.False(t, ok, "ext.Extract ok")
	require.Nil(t, authReq)
	require.EqualError(t, ext.FailureReason(h), stream.ErrInvalidAuthHeaders.Error())
}

func TestExtractor_Extract_InvalidToken(t *testing.T) {
	ext := NewExtractor()
	h := http.Header{"Authorization": {"Bearer foobar-not-a-jwt"}} // invalid JWT

	authReq, ok := ext.Extract(h)
	require.False(t, ok, "ext.Extract ok")
	require.Nil(t, authReq)
	require.EqualError(t, ext.FailureReason(h), stream.ErrInvalidJWT.Error())
}
