package s2s

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"

	"github.com/stretchr/testify/require"

	"github.com/stretchr/testify/assert"
)

func TestNewResolver(t *testing.T) {
	// valid
	res, err := NewResolver(newValidSecret(), "aud-you")
	assert.NoError(t, err)
	assert.NotNil(t, res)

	// invalid secret
	_, err = NewResolver(newSecretWithLen(0), "aud-you")
	assert.EqualError(t, err, "s2s-secret is empty or has less than 64 bytes")
}

func TestResolver_Method(t *testing.T) {
	res, err := NewResolver(newValidSecret(), "aud-you")
	assert.NoError(t, err)
	assert.Equal(t, s2sMethod, res.Method())
}

func TestResolver_Resolve_ReqWithValidEncodedToken(t *testing.T) {
	secret := newValidSecret()
	res, err := NewResolver(secret, "aud-you")
	require.NoError(t, err)

	req := newFakeRequest(t, secret, "aud-you")

	creds, err := res.Resolve(req)
	require.NoError(t, err)
	require.NotNil(t, creds)
}

func TestResolver_Resolve_ReqWithParsedJWT(t *testing.T) {
	secret := newValidSecret()
	res, err := NewResolver(secret, "aud-you")
	require.NoError(t, err)

	dec := NewDecoder()
	r := newFakeRequest(t, secret, "aud-you")

	decodedReq, ok := dec.Decode(r.encodedJWT)
	assert.True(t, ok, "dec.Decode ok")
	req := decodedReq.(*request)
	assert.NotNil(t, req.parsedJWT) // parsedJWT is present

	creds, err := res.Resolve(req)
	require.NoError(t, err)
	require.NotNil(t, creds)
}

func TestResolver_Resolve_ReqWithDifferentSecret(t *testing.T) {
	secret := newSecretWithLen(64)
	dec := NewDecoder()
	r := newFakeRequest(t, secret, "aud-you")
	decodedReq, ok := dec.Decode(r.encodedJWT)
	assert.True(t, ok, "dec.Decode ok")
	req := decodedReq.(*request)

	anotherSecret := newSecretWithLen(65)
	res, err := NewResolver(anotherSecret, "aud-you")
	require.NoError(t, err)

	_, err = res.Resolve(req)
	require.Equal(t, stream.ErrInvalidSignature, err)
}

func TestResolver_Resolve_ReqWithDifferentAudience(t *testing.T) {
	secret := newValidSecret()
	dec := NewDecoder()
	r := newFakeRequest(t, secret, "aud-you")
	decodedReq, ok := dec.Decode(r.encodedJWT)
	assert.True(t, ok, "dec.Decode ok")
	req := decodedReq.(*request)

	res, err := NewResolver(secret, "another-aud-them")
	require.NoError(t, err)

	_, err = res.Resolve(req)
	require.Equal(t, stream.ErrAuthAudienceMismatch, err)
}
